"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPartitionKeyNameFromTable = exports.buildDynamoDBTableWithStream = exports.checkTableProps = exports.buildDynamoDBTable = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const dynamodb_table_defaults_1 = require("./dynamodb-table-defaults");
const utils_1 = require("./utils");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildDynamoDBTable(scope, props) {
    checkTableProps(props);
    // Conditional DynamoDB Table creation
    if (props.existingTableObj) {
        return { tableInterface: props.existingTableObj, tableObject: props.existingTableObj };
    }
    else if (props.existingTableInterface) {
        return { tableInterface: props.existingTableInterface };
    }
    else {
        const consolidatedTableProps = utils_1.consolidateProps(dynamodb_table_defaults_1.DefaultTableProps, props.dynamoTableProps);
        const newTable = new dynamodb.Table(scope, 'DynamoTable', consolidatedTableProps);
        return { tableInterface: newTable, tableObject: newTable };
    }
}
exports.buildDynamoDBTable = buildDynamoDBTable;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function checkTableProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.dynamoTableProps && props.existingTableObj) {
        errorMessages += 'Error - Either provide existingTableObj or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (props.dynamoTableProps && props.existingTableInterface) {
        errorMessages += 'Error - Either provide existingTableInterface or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (props.existingTableObj && props.existingTableInterface) {
        errorMessages += 'Error - Either provide existingTableInterface or existingTableObj, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.checkTableProps = checkTableProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildDynamoDBTableWithStream(scope, props) {
    // Conditional DynamoDB Table creation
    if (!props.existingTableInterface) {
        // Set the default props for DynamoDB table
        const dynamoTableProps = utils_1.consolidateProps(dynamodb_table_defaults_1.DefaultTableWithStreamProps, props.dynamoTableProps);
        const dynamoTable = new dynamodb.Table(scope, 'DynamoTable', dynamoTableProps);
        return { tableInterface: dynamoTable, tableObject: dynamoTable };
    }
    else {
        return { tableInterface: props.existingTableInterface };
    }
}
exports.buildDynamoDBTableWithStream = buildDynamoDBTableWithStream;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function getPartitionKeyNameFromTable(table) {
    const cfnTable = table.node.findChild('Resource');
    const keySchema = cfnTable.keySchema;
    const partitionKey = keySchema.find((keyPart) => keyPart.keyType === 'HASH');
    if (!partitionKey) {
        throw new Error('Partition key for table not defined');
    }
    return partitionKey.attributeName;
}
exports.getPartitionKeyNameFromTable = getPartitionKeyNameFromTable;
//# sourceMappingURL=data:application/json;base64,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