"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCachePort = exports.createCacheSubnetGroup = exports.obtainMemcachedCluster = void 0;
const cache = require("aws-cdk-lib/aws-elasticache");
const elasticache_defaults_1 = require("./elasticache-defaults");
const utils_1 = require("./utils");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function obtainMemcachedCluster(scope, id, props) {
    if (props.existingCache) {
        props.existingCache.vpcSecurityGroupIds?.push(props.cacheSecurityGroupId);
        return props.existingCache;
    }
    else {
        if (!props.cachePort) {
            throw Error('props.cachePort required for new caches');
        }
        // Create the subnet group from all the isolated subnets in the VPC
        const subnetGroup = createCacheSubnetGroup(scope, props.vpc, id);
        const defaultProps = elasticache_defaults_1.GetMemcachedDefaults(id, props.cachePort);
        const requiredConstructProps = {
            vpcSecurityGroupIds: [props.cacheSecurityGroupId],
            cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
        };
        const consolidatedProps = utils_1.consolidateProps(defaultProps, props.cacheProps, requiredConstructProps, true);
        const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, consolidatedProps);
        newCache.addDependency(subnetGroup);
        return newCache;
    }
}
exports.obtainMemcachedCluster = obtainMemcachedCluster;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createCacheSubnetGroup(construct, vpc, id) {
    // Memcached has no auth, all access control is
    // network based, so, at least initially, we will
    // only launch it in isolated subnets.
    const subnetIds = [];
    vpc.isolatedSubnets.forEach((subnet) => {
        subnetIds.push(subnet.subnetId);
    });
    return new cache.CfnSubnetGroup(construct, `ec-subnetgroup-${id}`, {
        description: "Solutions Constructs generated Cache Subnet Group",
        subnetIds,
        cacheSubnetGroupName: `${id}-subnet-group`,
    });
}
exports.createCacheSubnetGroup = createCacheSubnetGroup;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function getCachePort(clientCacheProps, existingCache) {
    if (existingCache) {
        return existingCache.attrConfigurationEndpointPort;
    }
    else if (clientCacheProps?.port) {
        return clientCacheProps.port;
    }
    else {
        return elasticache_defaults_1.GetDefaultCachePort();
    }
}
exports.getCachePort = getCachePort;
//# sourceMappingURL=data:application/json;base64,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