"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const elasticsearch = require("aws-cdk-lib/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const vpc_helper_1 = require("./vpc-helper");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const MaximumAzsInElasticsearchDomain = 3;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildElasticSearch(scope, props) {
    let subnetIds = [];
    const constructDrivenProps = {};
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = createKibanaCognitoRole(scope, props.userpool, props.identitypool, props.domainName);
    if (props.vpc) {
        subnetIds = vpc_helper_1.retrievePrivateSubnetIds(props.vpc);
        if (subnetIds.length > MaximumAzsInElasticsearchDomain) {
            subnetIds = subnetIds.slice(0, MaximumAzsInElasticsearchDomain);
        }
        constructDrivenProps.vpcOptions = {
            subnetIds,
            securityGroupIds: props.securityGroupIds
        };
        // If the client did not submit a ClusterConfig, then we will create one
        if (!props.clientDomainProps?.elasticsearchClusterConfig) {
            constructDrivenProps.elasticsearchClusterConfig = createClusterConfiguration(subnetIds.length);
        }
    }
    else { // No VPC
        // If the client did not submit a ClusterConfig, then we will create one based on the Region
        if (!props.clientDomainProps?.elasticsearchClusterConfig) {
            constructDrivenProps.elasticsearchClusterConfig = createClusterConfiguration(cdk.Stack.of(scope).availabilityZones.length);
        }
    }
    const defaultCfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(props.domainName, cognitoKibanaConfigureRole, props);
    const finalCfnDomainProps = utils_1.consolidateProps(defaultCfnDomainProps, props.clientDomainProps, constructDrivenProps);
    const esDomain = new elasticsearch.CfnDomain(scope, `ElasticsearchDomain`, finalCfnDomainProps);
    utils_1.addCfnSuppressRules(esDomain, [
        {
            id: "W28",
            reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return { domain: esDomain, role: cognitoKibanaConfigureRole };
}
exports.buildElasticSearch = buildElasticSearch;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace',
            statistic: 'Minimum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
function createClusterConfiguration(numberOfAzs) {
    return {
        dedicatedMasterEnabled: true,
        dedicatedMasterCount: 3,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
            availabilityZoneCount: numberOfAzs
        },
        instanceCount: numberOfAzs,
    };
}
function createKibanaCognitoRole(scope, userPool, identitypool, domainName) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, "CognitoKibanaConfigureRole", {
        assumedBy: new iam.ServicePrincipal("es.amazonaws.com"),
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, "CognitoKibanaConfigureRolePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig",
                ],
                resources: [
                    userPool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["iam:PassRole"],
                conditions: {
                    StringLike: {
                        "iam:PassedToService": "cognito-identity.amazonaws.com",
                    },
                },
                resources: [cognitoKibanaConfigureRole.roleArn],
            }),
        ],
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    return cognitoKibanaConfigureRole;
}
//# sourceMappingURL=data:application/json;base64,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