"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getServiceVpcSecurityGroupIds = exports.CheckFargateProps = exports.CreateFargateService = void 0;
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const ecr = require("aws-cdk-lib/aws-ecr");
const defaults = require("..");
const __1 = require("..");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CreateFargateService(scope, id, constructVpc, clientClusterProps, ecrRepositoryArn, ecrImageVersion, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, clientFargateServiceProps) {
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.ECR_API);
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.ECR_DKR);
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.S3);
    const constructContainerDefintionProps = {};
    const constructFargateServiceDefinitionProps = {};
    if (!clientFargateServiceProps?.cluster) {
        // Construct Fargate Service
        constructFargateServiceDefinitionProps.cluster = CreateCluster(scope, `${id}-cluster`, constructVpc, clientClusterProps);
    }
    // Set up the Fargate service
    if (!clientContainerDefinitionProps?.image) {
        constructContainerDefintionProps.image = CreateImage(scope, id, ecrRepositoryArn, ecrImageVersion);
    }
    // Create the Fargate Service
    let newContainerDefinition;
    const createTaskDefinitionResponse = CreateTaskDefinition(scope, id, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, constructContainerDefintionProps);
    constructFargateServiceDefinitionProps.taskDefinition = createTaskDefinitionResponse.taskDefinition;
    newContainerDefinition = createTaskDefinitionResponse.containerDefinition;
    if (!clientFargateServiceProps?.vpcSubnets) {
        if (constructVpc.isolatedSubnets.length) {
            constructFargateServiceDefinitionProps.vpcSubnets = {
                subnets: constructVpc.isolatedSubnets,
            };
        }
        else {
            constructFargateServiceDefinitionProps.vpcSubnets = {
                subnets: constructVpc.privateSubnets,
            };
        }
    }
    let defaultFargateServiceProps;
    if (!clientFargateServiceProps?.securityGroups) {
        const serviceSecurityGroup = new ec2.SecurityGroup(scope, `${id}-sg`, {
            allowAllOutbound: true,
            disableInlineRules: false,
            vpc: constructVpc,
            // We add a description here so that this SG can be easily identified in tests
            description: 'Construct created security group'
        });
        defaultFargateServiceProps = __1.overrideProps(defaults.DefaultFargateServiceProps(), { securityGroups: [serviceSecurityGroup] });
        defaults.addCfnSuppressRules(serviceSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
    }
    else {
        defaultFargateServiceProps = defaults.DefaultFargateServiceProps();
    }
    const fargateServiceProps = defaults.consolidateProps(defaultFargateServiceProps, clientFargateServiceProps, constructFargateServiceDefinitionProps);
    const newService = new ecs.FargateService(scope, `${id}-service`, fargateServiceProps);
    return { service: newService, containerDefinition: newContainerDefinition };
}
exports.CreateFargateService = CreateFargateService;
function CreateCluster(scope, id, constructVpc, clientClusterProps) {
    const constructProps = { vpc: constructVpc };
    return new ecs.Cluster(scope, id, defaults.consolidateProps(defaults.DefaultClusterProps(), clientClusterProps, constructProps));
}
function CreateImage(scope, id, repositoryArn, imageVersion) {
    if (repositoryArn) {
        return ecs.ContainerImage.fromEcrRepository(ecr.Repository.fromRepositoryArn(scope, `${id}-container`, repositoryArn), imageVersion || "latest");
    }
    else {
        throw new Error("Not Implemented - image without repo name and version");
    }
}
function CreateTaskDefinition(scope, id, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, constructContainerDefinitionProps) {
    const taskDefinitionProps = defaults.consolidateProps(defaults.DefaultFargateTaskDefinitionProps(), clientFargateTaskDefinitionProps);
    const taskDefinition = new ecs.FargateTaskDefinition(scope, `${id}-taskdef`, taskDefinitionProps);
    const defaultContainerDefinitionProps = defaults.consolidateProps(defaults.DefaultContainerDefinitionProps(), {
        containerName: `${id}-container`,
    });
    const containerDefinitionProps = defaults.consolidateProps(defaultContainerDefinitionProps, clientContainerDefinitionProps, constructContainerDefinitionProps);
    const containerDefinition = taskDefinition.addContainer(`${id}-container`, containerDefinitionProps);
    return { taskDefinition, containerDefinition };
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckFargateProps(props) {
    let errorMessages = "";
    let errorFound = false;
    if (props.existingFargateServiceObject &&
        (props.existingImageObject ||
            props.ecrImageVersion ||
            props.containerDefinitionProps ||
            props.fargateTaskDefinitionProps ||
            props.ecrRepositoryArn ||
            props.fargateServiceProps ||
            props.clusterProps)) {
        errorFound = true;
        errorMessages +=
            "If you provide an existingFargateServiceObject, you cannot provide any props defining a new service\n";
    }
    if (props.existingImageObject &&
        (props.ecrImageVersion || props.ecrRepositoryArn)) {
        errorFound = true;
        errorMessages +=
            "If you provide an existingImageObject then you cannot provide an ecrRepositoryArn nor ecrImageVersion\n";
    }
    // Confirm no network information in Target Group Props
    if (props.targetGroupProps?.vpc) {
        errorFound = true;
        errorMessages +=
            "Provide all VPC info at Construct level, not within targetGroupProps\n";
    }
    if (props.fargateServiceProps?.taskDefinition) {
        errorFound = true;
        errorMessages +=
            "The construct cannot accept an existing task definition in fargateServiceProps\n";
    }
    if (props.fargateServiceProps?.cluster && props.clusterProps) {
        errorFound = true;
        errorMessages +=
            "If you provide a cluster in fargateServiceProps then you cannot provide clusterProps\n";
    }
    if (props.clusterProps?.vpc) {
        errorFound = true;
        errorMessages +=
            "All services in the construct use the construct VPC, you cannot specify a VPC in clusterProps\n";
    }
    if ((props.existingFargateServiceObject ||
        props.existingContainerDefinitionObject) &&
        (!props.existingFargateServiceObject ||
            !props.existingContainerDefinitionObject ||
            !props.existingVpc)) {
        errorFound = true;
        errorMessages +=
            "If an existing Service is indicated by supplying either existingFargateServiceObject or existingContainerDefinitionObject, then existingFargateServiceObject, existingContainerDefinitionObject, and existingVpc must all be provided\n";
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckFargateProps = CheckFargateProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function getServiceVpcSecurityGroupIds(service) {
    const securityGroupIds = [];
    service.connections.securityGroups.forEach(element => securityGroupIds.push(element.securityGroupId));
    return securityGroupIds;
}
exports.getServiceVpcSecurityGroupIds = getServiceVpcSecurityGroupIds;
//# sourceMappingURL=data:application/json;base64,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