"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            const deployGlueJobResponse = deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore, props.etlCodeAsset);
            return {
                job: deployGlueJobResponse.job,
                role: deployGlueJobResponse.role,
                bucket: deployGlueJobResponse.bucket,
                loggingBucket: deployGlueJobResponse.loggingBucket
            };
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return { job: props.existingCfnJob, role: aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role) };
    }
}
exports.buildGlueJob = buildGlueJob;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore, etlCodeAsset) {
    let glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        glueSecurityConfigName = 'ETLJobSecurityConfig';
        const glueKMSKey = `arn:${aws_cdk_lib_1.Aws.PARTITION}:kms:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    const jobRole = glueJobProps.role ?
        aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role) :
        defaults.createGlueJobRole(scope);
    glueJobPolicy.attachToRole(jobRole);
    let outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            outputLocation = { bucket: outputDataStore.existingS3OutputBucket };
        }
        else {
            outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        outputLocation = defaults.buildS3Bucket(scope, {});
    }
    outputLocation.bucket.grantReadWrite(jobRole);
    const jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${outputLocation.bucket.bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(jobRole, glueJobProps, glueSecurityConfigName, jobArgumentsList, etlCodeAsset), glueJobProps);
    if (etlCodeAsset) {
        etlCodeAsset.grantRead(jobRole);
    }
    else {
        // create CDK Bucket instance from S3 url and grant read access to Glue Job's service principal
        if (isJobCommandProperty(newGlueJobProps.command)) {
            if (!newGlueJobProps.command.scriptLocation) {
                throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
            }
            const scriptLocation = newGlueJobProps.command.scriptLocation;
            const scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(scriptLocation));
            scriptBucketLocation.grantRead(jobRole);
        }
    }
    const glueJob = new glue.CfnJob(scope, 'KinesisETLJob', newGlueJobProps);
    return { job: glueJob, role: jobRole, bucket: outputLocation.bucket, loggingBucket: outputLocation.loggingBucket };
}
exports.deployGlueJob = deployGlueJob;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    if (s3Url && s3Url.startsWith('s3://')) {
        const splitString = s3Url.slice('s3://'.length);
        return `arn:${aws_cdk_lib_1.Aws.PARTITION}:s3:::${splitString}`;
    }
    else {
        throw Error(`Received S3URL as ${s3Url}. The S3 url string does not begin with s3://. This is not a standard S3 url`);
    }
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type. For the construct to work for streaming ETL from Kinesis Data
 * Streams, all three attributes of the JobCommandProperty are required, even though they may be optional for other use cases.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name &&
        command.pythonVersion &&
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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