"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckListValues = exports.CheckProps = void 0;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckProps(propsObject) {
    let errorMessages = '';
    let errorFound = false;
    if (propsObject.loadBalancerProps && propsObject.existingLoadBalancerObj) {
        errorMessages += 'Error - Either provide loadBalancerProps or existingLoadBalancerObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.dynamoTableProps && propsObject.existingTableObj) {
        errorMessages += 'Error - Either provide existingTableObj or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.dynamoTableProps && propsObject.existingTableInterface) {
        errorMessages += 'Error - Either provide existingTableInterface or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingStreamObj && propsObject.kinesisStreamProps) {
        errorMessages += 'Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingLambdaObj && propsObject.lambdaFunctionProps) {
        errorMessages += 'Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingQueueObj && propsObject.queueProps) {
        errorMessages += 'Error - Either provide queueProps or existingQueueObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.queueProps?.encryptionMasterKey && propsObject.encryptionKey) {
        errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or encryptionKey, but not both.\n';
        errorFound = true;
    }
    if (propsObject.queueProps?.encryptionMasterKey && propsObject.encryptionKeyProps) {
        errorMessages += 'Error - Either provide queueProps.encryptionMasterKey or encryptionKeyProps, but not both.\n';
        errorFound = true;
    }
    if ((propsObject?.deployDeadLetterQueue === false) && propsObject.deadLetterQueueProps) {
        errorMessages += 'Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n';
        errorFound = true;
    }
    const isQueueFifo = propsObject?.queueProps?.fifo;
    const isDeadLetterQueueFifo = propsObject?.deadLetterQueueProps?.fifo;
    const deployDeadLetterQueue = propsObject.deployDeadLetterQueue || propsObject.deployDeadLetterQueue === undefined;
    if (deployDeadLetterQueue && (isQueueFifo !== isDeadLetterQueueFifo)) {
        errorMessages += 'Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n';
        errorFound = true;
    }
    if (propsObject.existingMediaStoreContainerObj && propsObject.mediaStoreContainerProps) {
        errorMessages += 'Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingBucketObj && propsObject.bucketProps) {
        errorMessages += 'Error - Either provide bucketProps or existingBucketObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingBucketInterface && propsObject.bucketProps) {
        errorMessages += 'Error - Either provide bucketProps or existingBucketInterface, but not both.\n';
        errorFound = true;
    }
    if (propsObject.topicProps && propsObject.existingTopicObj) {
        errorMessages += 'Error - Either provide topicProps or existingTopicObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.topicProps?.masterKey && propsObject.encryptionKey) {
        errorMessages += 'Error - Either provide topicProps.masterKey or encryptionKey, but not both.\n';
        errorFound = true;
    }
    if (propsObject.topicProps?.masterKey && propsObject.encryptionKeyProps) {
        errorMessages += 'Error - Either provide topicProps.masterKey or encryptionKeyProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.glueJobProps && propsObject.existingGlueJob) {
        errorMessages += 'Error - Either provide glueJobProps or existingGlueJob, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingSagemakerEndpointObj && propsObject.endpointProps) {
        errorMessages += 'Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingSecretObj && propsObject.secretProps) {
        errorMessages += 'Error - Either provide secretProps or existingSecretObj, but not both.\n';
        errorFound = true;
    }
    if ((propsObject.deployVpc || propsObject.vpcProps) && propsObject.existingVpc) {
        errorMessages += 'Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.encryptionKey && propsObject.encryptionKeyProps) {
        errorMessages += 'Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingEventBusInterface && propsObject.eventBusProps) {
        errorMessages += 'Error - Either provide existingEventBusInterface or eventBusProps, but not both.\n';
        errorFound = true;
    }
    if (propsObject.existingWebaclObj && propsObject.webaclProps) {
        errorMessages += 'Error - Either provide existingWebaclObj or webaclProps, but not both.\n';
        errorFound = true;
    }
    if ((propsObject?.logAlbAccessLogs === false) && (propsObject.albLoggingBucketProps)) {
        errorMessages += 'Error - If logAlbAccessLogs is false, supplying albLoggingBucketProps is invalid.\n';
        errorFound = true;
    }
    if (propsObject.existingLoggingBucketObj && propsObject.loggingBucketProps) {
        errorMessages += 'Error - Either provide existingLoggingBucketObj or loggingBucketProps, but not both.\n';
        errorFound = true;
    }
    if ((propsObject?.logS3AccessLogs === false) && (propsObject.loggingBucketProps || propsObject.existingLoggingBucketObj)) {
        errorMessages += 'Error - If logS3AccessLogs is false, supplying loggingBucketProps or existingLoggingBucketObj is invalid.\n';
        errorFound = true;
    }
    if (propsObject.existingBucketObj && (propsObject.loggingBucketProps || propsObject.logS3AccessLogs)) {
        errorMessages += 'Error - If existingBucketObj is provided, supplying loggingBucketProps or logS3AccessLogs is an error.\n';
        errorFound = true;
    }
    if (propsObject.insertHttpSecurityHeaders !== false && propsObject.responseHeadersPolicyProps?.securityHeadersBehavior) {
        errorMessages += 'responseHeadersPolicyProps.securityHeadersBehavior can only be passed if httpSecurityHeaders is set to `false`.';
        errorFound = true;
    }
    if (propsObject.openSearchDomainProps?.vpcOptions) {
        throw new Error("Error - Define VPC using construct parameters not the OpenSearch Service props");
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckProps = CheckProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function CheckListValues(allowedPermissions, submittedValues, valueType) {
    submittedValues.forEach((submittedValue) => {
        if (!allowedPermissions.includes(submittedValue)) {
            throw Error(`Invalid ${valueType} submitted - ${submittedValue}`);
        }
    });
}
exports.CheckListValues = CheckListValues;
//# sourceMappingURL=data:application/json;base64,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