"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
// Imports
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    // If property overrides have been provided, incorporate them and deploy
    kinesisStreamProps = utils_1.consolidateProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds',
            statistic: 'Maximum',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 43200000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded',
            statistic: 'Average',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 0,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy1zdHJlYW1zLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImtpbmVzaXMtc3RyZWFtcy1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFFSDs7O0dBR0c7QUFFSCxVQUFVO0FBQ1YsbURBQW1EO0FBQ25ELHlFQUFnRTtBQUNoRSxtQ0FBbUM7QUFDbkMsbUNBQTJDO0FBQzNDLHlEQUF5RDtBQW9CekQ7O0dBRUc7QUFDSCxTQUFnQixrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEtBQThCO0lBRWpGLElBQUksS0FBSyxDQUFDLGlCQUFpQixFQUFFO1FBQzNCLE9BQU8sS0FBSyxDQUFDLGlCQUFpQixDQUFDO0tBQ2hDO0lBRUQsOEJBQThCO0lBQzlCLElBQUksa0JBQWtCLENBQUM7SUFDdkIsd0VBQXdFO0lBQ3hFLGtCQUFrQixHQUFHLHdCQUFnQixDQUFDLDZDQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBRXBGLCtCQUErQjtJQUMvQixPQUFPLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLGtCQUFrQixDQUFDLENBQUM7QUFDeEUsQ0FBQztBQWJELGdEQWFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQiwwQkFBMEIsQ0FBQyxLQUFnQjtJQUN6RCxvQ0FBb0M7SUFDcEMsTUFBTSxNQUFNLEdBQXVCLElBQUksS0FBSyxFQUFFLENBQUM7SUFFL0MsNEdBQTRHO0lBQzVHLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSx5Q0FBeUMsRUFBRTtRQUNqRixNQUFNLEVBQUUsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzVCLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLFVBQVUsRUFBRSxvQ0FBb0M7WUFDaEQsU0FBUyxFQUFFLFNBQVM7WUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUNoQyxDQUFDO1FBQ0YsU0FBUyxFQUFFLFFBQVE7UUFDbkIsaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLGtHQUFrRztLQUNySCxDQUFDLENBQUMsQ0FBQztJQUVKLHdEQUF3RDtJQUN4RCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUscURBQXFELEVBQUU7UUFDN0YsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsYUFBYTtZQUN4QixVQUFVLEVBQUUsbUNBQW1DO1lBQy9DLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDaEMsQ0FBQztRQUNGLFNBQVMsRUFBRSxDQUFDO1FBQ1osaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1FBQ3hFLGdCQUFnQixFQUFFLGlFQUFpRTtLQUNwRixDQUFDLENBQUMsQ0FBQztJQUVKLE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFqQ0QsZ0VBaUNDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLypcbiAqICBUaGUgZnVuY3Rpb25zIGZvdW5kIGhlcmUgaW4gdGhlIGNvcmUgbGlicmFyeSBhcmUgZm9yIGludGVybmFsIHVzZSBhbmQgY2FuIGJlIGNoYW5nZWRcbiAqICBvciByZW1vdmVkIG91dHNpZGUgb2YgYSBtYWpvciByZWxlYXNlLiBXZSByZWNvbW1lbmQgYWdhaW5zdCBjYWxsaW5nIHRoZW0gZGlyZWN0bHkgZnJvbSBjbGllbnQgY29kZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgKiBhcyBraW5lc2lzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1raW5lc2lzJztcbmltcG9ydCB7IERlZmF1bHRTdHJlYW1Qcm9wcyB9IGZyb20gJy4va2luZXNpcy1zdHJlYW1zLWRlZmF1bHRzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBjb25zb2xpZGF0ZVByb3BzIH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoJztcbi8vIE5vdGU6IFRvIGVuc3VyZSBDREt2MiBjb21wYXRpYmlsaXR5LCBrZWVwIHRoZSBpbXBvcnQgc3RhdGVtZW50IGZvciBDb25zdHJ1Y3Qgc2VwYXJhdGVcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkS2luZXNpc1N0cmVhbVByb3BzIHtcbiAgLyoqXG4gICAqIEV4aXN0aW5nIGluc3RhbmNlIG9mIEtpbmVzaXMgU3RyZWFtLCBwcm92aWRpbmcgYm90aCB0aGlzIGFuZCBga2luZXNpc1N0cmVhbVByb3BzYCB3aWxsIGNhdXNlIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nU3RyZWFtT2JqPzoga2luZXNpcy5TdHJlYW07XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIHVzZXIgcHJvdmlkZWQgcHJvcHMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcHMgZm9yIHRoZSBLaW5lc2lzIHN0cmVhbS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkga2luZXNpc1N0cmVhbVByb3BzPzoga2luZXNpcy5TdHJlYW1Qcm9wc1xufVxuXG4vKipcbiAqIEBpbnRlcm5hbCBUaGlzIGlzIGFuIGludGVybmFsIGNvcmUgZnVuY3Rpb24gYW5kIHNob3VsZCBub3QgYmUgY2FsbGVkIGRpcmVjdGx5IGJ5IFNvbHV0aW9ucyBDb25zdHJ1Y3RzIGNsaWVudHMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBidWlsZEtpbmVzaXNTdHJlYW0oc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM6IEJ1aWxkS2luZXNpc1N0cmVhbVByb3BzKToga2luZXNpcy5TdHJlYW0ge1xuXG4gIGlmIChwcm9wcy5leGlzdGluZ1N0cmVhbU9iaikge1xuICAgIHJldHVybiBwcm9wcy5leGlzdGluZ1N0cmVhbU9iajtcbiAgfVxuXG4gIC8vIFNldHVwIHRoZSBzdHJlYW0gcHJvcGVydGllc1xuICBsZXQga2luZXNpc1N0cmVhbVByb3BzO1xuICAvLyBJZiBwcm9wZXJ0eSBvdmVycmlkZXMgaGF2ZSBiZWVuIHByb3ZpZGVkLCBpbmNvcnBvcmF0ZSB0aGVtIGFuZCBkZXBsb3lcbiAga2luZXNpc1N0cmVhbVByb3BzID0gY29uc29saWRhdGVQcm9wcyhEZWZhdWx0U3RyZWFtUHJvcHMsIHByb3BzLmtpbmVzaXNTdHJlYW1Qcm9wcyk7XG5cbiAgLy8gQ3JlYXRlIHRoZSBzdHJlYW0gYW5kIHJldHVyblxuICByZXR1cm4gbmV3IGtpbmVzaXMuU3RyZWFtKHNjb3BlLCAnS2luZXNpc1N0cmVhbScsIGtpbmVzaXNTdHJlYW1Qcm9wcyk7XG59XG5cbi8qKlxuICogQGludGVybmFsIFRoaXMgaXMgYW4gaW50ZXJuYWwgY29yZSBmdW5jdGlvbiBhbmQgc2hvdWxkIG5vdCBiZSBjYWxsZWQgZGlyZWN0bHkgYnkgU29sdXRpb25zIENvbnN0cnVjdHMgY2xpZW50cy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkS2luZXNpc1N0cmVhbUNXQWxhcm1zKHNjb3BlOiBDb25zdHJ1Y3QpOiBjbG91ZHdhdGNoLkFsYXJtW10ge1xuICAvLyBTZXR1cCBDVyBBbGFybXMgZm9yIEtpbmVzaXNTdHJlYW1cbiAgY29uc3QgYWxhcm1zOiBjbG91ZHdhdGNoLkFsYXJtW10gPSBuZXcgQXJyYXkoKTtcblxuICAvLyBBbGFybSBpZiBNYXggKEdldFJlY29yZHMuSXRlcmF0b3JBZ2VNaWxsaXNlY29uZHMpOiA+PSAxMiBob3VycyAoNTAlIG9mIDI0IGhvdXJzIGRlZmF1bHQgcmV0ZW50aW9uIHBlcmlvZClcbiAgYWxhcm1zLnB1c2gobmV3IGNsb3Vkd2F0Y2guQWxhcm0oc2NvcGUsICdLaW5lc2lzU3RyZWFtR2V0UmVjb3Jkc0l0ZXJhdG9yQWdlQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0tpbmVzaXMnLFxuICAgICAgbWV0cmljTmFtZTogJ0dldFJlY29yZHMuSXRlcmF0b3JBZ2VNaWxsaXNlY29uZHMnLFxuICAgICAgc3RhdGlzdGljOiAnTWF4aW11bScsXG4gICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDUpLFxuICAgIH0pLFxuICAgIHRocmVzaG9sZDogNDMyMDAwMDAsIC8vIDQzMjAwMDAwIE1pbGxpc2Vjb25kcyA9IDEyIEhvdXJzICg1MCUgb2YgMjQgaG91cnMgLSBkZWZhdWx0IHJlY29yZCByZXRlbnRpb24gcGVyaW9kKVxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQ29uc3VtZXIgUmVjb3JkIFByb2Nlc3NpbmcgRmFsbGluZyBCZWhpbmQsIHRoZXJlIGlzIHJpc2sgZm9yIGRhdGEgbG9zcyBkdWUgdG8gcmVjb3JkIGV4cGlyYXRpb24uJ1xuICB9KSk7XG5cbiAgLy8gQWxhcm0gaWYgQXZnIChSZWFkUHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWQpOiA+IDBcbiAgYWxhcm1zLnB1c2gobmV3IGNsb3Vkd2F0Y2guQWxhcm0oc2NvcGUsICdLaW5lc2lzU3RyZWFtUmVhZFByb3Zpc2lvbmVkVGhyb3VnaHB1dEV4Y2VlZGVkQWxhcm0nLCB7XG4gICAgbWV0cmljOiBuZXcgY2xvdWR3YXRjaC5NZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0tpbmVzaXMnLFxuICAgICAgbWV0cmljTmFtZTogJ1JlYWRQcm92aXNpb25lZFRocm91Z2hwdXRFeGNlZWRlZCcsXG4gICAgICBzdGF0aXN0aWM6ICdBdmVyYWdlJyxcbiAgICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgfSksXG4gICAgdGhyZXNob2xkOiAwLFxuICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQ29uc3VtZXIgQXBwbGljYXRpb24gaXMgUmVhZGluZyBhdCBhIFNsb3dlciBSYXRlIFRoYW4gRXhwZWN0ZWQuJ1xuICB9KSk7XG5cbiAgcmV0dXJuIGFsYXJtcztcbn0iXX0=