"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLambdaVpcSecurityGroupIds = exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, props.lambdaFunctionProps.functionName, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            const levelOneFunction = props.existingLambdaObj.node.defaultChild;
            if (props.lambdaFunctionProps?.securityGroups) {
                let ctr = 20;
                props.lambdaFunctionProps?.securityGroups.forEach(sg => {
                    // TODO: Discuss with someone why I can't get R/O access to VpcConfigSecurityGroupIds
                    levelOneFunction.addOverride(`Properties.VpcConfig.SecurityGroupIds.${ctr++}`, sg.securityGroupId);
                });
            }
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        }, true);
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    utils_1.addCfnSuppressRules(lambdafunction, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = lambdafunction.role?.node.tryFindChild('DefaultPolicy')?.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        utils_1.addCfnSuppressRules(cfnLambdafunctionDefPolicy, [
            {
                id: 'W12',
                reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
            }
        ]);
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * A wrapper above Function.addPermision that
 * prevents two different calls to addPermission using
 * the same construct id.
 */
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function getLambdaVpcSecurityGroupIds(lambdaFunction) {
    const securityGroupIds = [];
    lambdaFunction.connections.securityGroups.forEach(element => securityGroupIds.push(element.securityGroupId));
    return securityGroupIds;
}
exports.getLambdaVpcSecurityGroupIds = getLambdaVpcSecurityGroupIds;
//# sourceMappingURL=data:application/json;base64,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