"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultOpenSearchCfnDomainProps = void 0;
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function DefaultOpenSearchCfnDomainProps(domainName, cognitoConfigureRole, props) {
    const roleARNs = [];
    roleARNs.push(new iam.ArnPrincipal(props.cognitoAuthorizedRoleARN));
    if (props.serviceRoleARN) {
        roleARNs.push(new iam.ArnPrincipal(props.serviceRoleARN));
    }
    // Features supported by engine version:
    // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
    // https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_opensearchservice.CfnDomainProps.html
    return {
        domainName,
        engineVersion: 'OpenSearch_1.3',
        encryptionAtRestOptions: {
            enabled: true
        },
        nodeToNodeEncryptionOptions: {
            enabled: true
        },
        snapshotOptions: {
            automatedSnapshotStartHour: 1
        },
        ebsOptions: {
            ebsEnabled: true,
            volumeSize: 10
        },
        cognitoOptions: {
            enabled: true,
            identityPoolId: props.identitypool.ref,
            userPoolId: props.userpool.userPoolId,
            roleArn: cognitoConfigureRole.roleArn
        },
        accessPolicies: new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    principals: roleARNs,
                    actions: [
                        'es:ESHttp*'
                    ],
                    resources: [
                        `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}/*`
                    ]
                })
            ]
        })
    };
}
exports.DefaultOpenSearchCfnDomainProps = DefaultOpenSearchCfnDomainProps;
//# sourceMappingURL=data:application/json;base64,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