"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildOpenSearchCWAlarms = exports.buildOpenSearch = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const opensearch = require("aws-cdk-lib/aws-opensearchservice");
const opensearch_defaults_1 = require("./opensearch-defaults");
const vpc_helper_1 = require("./vpc-helper");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const MAXIMUM_AZS_IN_OPENSEARCH_DOMAIN = 3;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildOpenSearch(scope, props) {
    let subnetIds = [];
    const constructDrivenProps = {};
    // Setup the IAM Role & policy for the OpenSearch Service to configure Cognito User pool and Identity pool
    const cognitoDashboardConfigureRole = createDashboardCognitoRole(scope, props.userpool, props.identitypool, props.openSearchDomainName);
    if (props.vpc) {
        subnetIds = vpc_helper_1.retrievePrivateSubnetIds(props.vpc);
        if (subnetIds.length > MAXIMUM_AZS_IN_OPENSEARCH_DOMAIN) {
            subnetIds = subnetIds.slice(0, MAXIMUM_AZS_IN_OPENSEARCH_DOMAIN);
        }
        constructDrivenProps.vpcOptions = {
            subnetIds,
            securityGroupIds: props.securityGroupIds
        };
        // If the client did not submit a ClusterConfig, then we will create one
        if (!props.clientDomainProps?.clusterConfig) {
            constructDrivenProps.clusterConfig = createClusterConfiguration(subnetIds.length);
        }
    }
    else { // No VPC
        // If the client did not submit a ClusterConfig, then we will create one based on the Region
        if (!props.clientDomainProps?.clusterConfig) {
            constructDrivenProps.clusterConfig = createClusterConfiguration(cdk.Stack.of(scope).availabilityZones.length);
        }
    }
    const defaultCfnDomainProps = opensearch_defaults_1.DefaultOpenSearchCfnDomainProps(props.openSearchDomainName, cognitoDashboardConfigureRole, props);
    const finalCfnDomainProps = utils_1.consolidateProps(defaultCfnDomainProps, props.clientDomainProps, constructDrivenProps);
    const opensearchDomain = new opensearch.CfnDomain(scope, `OpenSearchDomain`, finalCfnDomainProps);
    utils_1.addCfnSuppressRules(opensearchDomain, [
        {
            id: "W28",
            reason: `The OpenSearch Service domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific OpenSearch Service instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return { domain: opensearchDomain, role: cognitoDashboardConfigureRole };
}
exports.buildOpenSearch = buildOpenSearch;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildOpenSearchCWAlarms(scope) {
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace',
            statistic: 'Minimum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildOpenSearchCWAlarms = buildOpenSearchCWAlarms;
function createClusterConfiguration(numberOfAzs) {
    return {
        dedicatedMasterEnabled: true,
        dedicatedMasterCount: 3,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
            availabilityZoneCount: numberOfAzs
        },
        instanceCount: numberOfAzs,
    };
}
function createDashboardCognitoRole(scope, userPool, identitypool, domainName) {
    // Setup the IAM Role & policy for the OpenSearch Service to configure Cognito User pool and Identity pool
    const cognitoDashboardConfigureRole = new iam.Role(scope, "CognitoDashboardConfigureRole", {
        assumedBy: new iam.ServicePrincipal("es.amazonaws.com"),
    });
    const cognitoDashboardConfigureRolePolicy = new iam.Policy(scope, "CognitoDashboardConfigureRolePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateDomainConfig",
                ],
                resources: [
                    userPool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["iam:PassRole"],
                conditions: {
                    StringLike: {
                        "iam:PassedToService": "cognito-identity.amazonaws.com",
                    },
                },
                resources: [cognitoDashboardConfigureRole.roleArn],
            }),
        ],
    });
    cognitoDashboardConfigureRolePolicy.attachToRole(cognitoDashboardConfigureRole);
    return cognitoDashboardConfigureRole;
}
//# sourceMappingURL=data:application/json;base64,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