"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.buildS3Bucket = exports.createAlbLoggingBucket = exports.createLoggingBucket = void 0;
const s3 = require("aws-cdk-lib/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createAlbLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: "This is a log bucket for an Application Load Balancer"
        }
    ]);
    return loggingBucket;
}
exports.createAlbLoggingBucket = createAlbLoggingBucket;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildS3Bucket(scope, props, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: aws_s3_1.StorageClass.GLACIER,
                    transitionAfter: aws_cdk_lib_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let customBucketProps;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    // If logging S3 access logs is enabled/undefined and an existing bucket object is not provided
    if (props.logS3AccessLogs !== false && !(props.bucketProps?.serverAccessLogsBucket)) {
        // Create the Logging Bucket
        let loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
        if (props.loggingBucketProps) {
            // User provided logging bucket props
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, props.loggingBucketProps);
        }
        else if (props.bucketProps?.removalPolicy) {
            // If the client explicitly specified a removal policy for the main bucket,
            // then replicate that policy on the logging bucket
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, { removalPolicy: props.bucketProps.removalPolicy });
        }
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, loggingBucketProps);
    }
    // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
    if (props.bucketProps?.versioned === undefined || props.bucketProps.versioned) {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
    }
    else {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    customBucketProps = props.bucketProps ? utils_1.overrideProps(customBucketProps, props.bucketProps) : customBucketProps;
    const s3Bucket = new s3.Bucket(scope, _bucketId, customBucketProps);
    return { bucket: s3Bucket, loggingBucket };
}
exports.buildS3Bucket = buildS3Bucket;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtYnVja2V0LWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInMzLWJ1Y2tldC1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFTSCx5Q0FBeUM7QUFFekMsNkRBQXNEO0FBQ3RELG1DQUE2RDtBQUM3RCwrQ0FBa0Q7QUFDbEQsNkNBQXVDO0FBMEJ2Qzs7R0FFRztBQUNILFNBQWdCLG1CQUFtQixDQUFDLEtBQWdCLEVBQ2xELFFBQWdCLEVBQ2hCLGtCQUFrQztJQUVsQyw0QkFBNEI7SUFDNUIsTUFBTSxhQUFhLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztJQUVwRiwrQ0FBK0M7SUFDL0MsTUFBTSxxQkFBcUIsR0FBRyxhQUFhLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQWlCLENBQUM7SUFFdkYsK0VBQStFO0lBQy9FLHFCQUFxQixDQUFDLG1CQUFtQixDQUFDLGVBQWUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO0lBRS9FLG1FQUFtRTtJQUNuRSxxQkFBcUIsQ0FBQywyQkFBMkIsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBRWxGLElBQUksT0FBTyxHQUFHLHdFQUF3RSxDQUFDO0lBRXZGLElBQUksUUFBUSxLQUFLLHlCQUF5QixFQUFFO1FBQzFDLE9BQU8sR0FBRyxpRkFBaUYsQ0FBQztLQUM3RjtJQUVELDJCQUFtQixDQUFDLHFCQUFxQixFQUFFO1FBQ3pDO1lBQ0UsRUFBRSxFQUFFLEtBQUs7WUFDVCxNQUFNLEVBQUUsT0FBTztTQUNoQjtLQUNGLENBQUMsQ0FBQztJQUVILE9BQU8sYUFBYSxDQUFDO0FBQ3ZCLENBQUM7QUE5QkQsa0RBOEJDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixzQkFBc0IsQ0FBQyxLQUFnQixFQUNyRCxRQUFnQixFQUNoQixrQkFBa0M7SUFFbEMsNEJBQTRCO0lBQzVCLE1BQU0sYUFBYSxHQUFjLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFFcEYsK0NBQStDO0lBQy9DLE1BQU0scUJBQXFCLEdBQUcsYUFBYSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFpQixDQUFDO0lBRXZGLDJCQUFtQixDQUFDLHFCQUFxQixFQUFFO1FBQ3pDO1lBQ0UsRUFBRSxFQUFFLEtBQUs7WUFDVCxNQUFNLEVBQUUsdURBQXVEO1NBQ2hFO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTyxhQUFhLENBQUM7QUFDdkIsQ0FBQztBQWxCRCx3REFrQkM7QUFPRDs7R0FFRztBQUNILFNBQWdCLGFBQWEsQ0FBQyxLQUFnQixFQUM1QyxLQUF5QixFQUN6QixRQUFpQjtJQUVqQixzRkFBc0Y7SUFDdEYsTUFBTSxjQUFjLEdBQXVCLENBQUM7WUFDMUMsNEJBQTRCLEVBQUUsQ0FBQztvQkFDN0IsWUFBWSxFQUFFLHFCQUFZLENBQUMsT0FBTztvQkFDbEMsZUFBZSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztpQkFDbkMsQ0FBQztTQUNILENBQUMsQ0FBQztJQUVILGdDQUFnQztJQUNoQyxJQUFJLGlCQUFpQyxDQUFDO0lBQ3RDLElBQUksYUFBYSxDQUFDO0lBQ2xCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsUUFBUSxHQUFHLFVBQVUsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDO0lBQ2hFLE1BQU0sZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLENBQUMsQ0FBQyxRQUFRLEdBQUcsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDO0lBRXJGLCtGQUErRjtJQUMvRixJQUFJLEtBQUssQ0FBQyxlQUFlLEtBQUssS0FBSyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLHNCQUFzQixDQUFDLEVBQUU7UUFDbkYsNEJBQTRCO1FBQzVCLElBQUksa0JBQWtCLEdBQUcsbUNBQWMsRUFBRSxDQUFDO1FBRTFDLElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzVCLHFDQUFxQztZQUNyQyxrQkFBa0IsR0FBRyxxQkFBYSxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQ2xGO2FBQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFLGFBQWEsRUFBRTtZQUMzQywyRUFBMkU7WUFDM0UsbURBQW1EO1lBQ25ELGtCQUFrQixHQUFHLHFCQUFhLENBQUMsa0JBQWtCLEVBQUUsRUFBRSxhQUFhLEVBQUUsS0FBSyxDQUFDLFdBQVcsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO1NBQzVHO1FBRUQsYUFBYSxHQUFHLG1CQUFtQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO0tBQ2xGO0lBQ0QseUVBQXlFO0lBQ3pFLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRSxTQUFTLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUyxFQUFFO1FBQzdFLGlCQUFpQixHQUFHLG1DQUFjLENBQUMsYUFBYSxFQUFFLGNBQWMsQ0FBQyxDQUFDO0tBQ25FO1NBQU07UUFDTCxpQkFBaUIsR0FBRyxtQ0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0tBQ25EO0lBRUQsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMscUJBQWEsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDO0lBRWhILE1BQU0sUUFBUSxHQUFjLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLGlCQUFpQixDQUFFLENBQUM7SUFFaEYsT0FBTyxFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsYUFBYSxFQUFFLENBQUM7QUFDN0MsQ0FBQztBQTlDRCxzQ0E4Q0M7QUFFRDs7R0FFRztBQUNILFNBQWdCLDRDQUE0QyxDQUFDLFNBQW9CLEVBQUUsU0FBaUI7SUFDbEcsTUFBTSw0QkFBNEIsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQW9CLENBQUM7SUFDL0YsTUFBTSxvQ0FBb0MsR0FBRyw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBYSxDQUFDO0lBQzdHLE1BQU0sc0NBQXNDLEdBQUcsb0NBQW9DLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQWUsQ0FBQztJQUVsSSw0Q0FBNEM7SUFDNUMsTUFBTSxVQUFVLEdBQUcsNEJBQTRCLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQXVCLENBQUM7SUFDakcsMkJBQW1CLENBQUMsVUFBVSxFQUFFO1FBQzlCO1lBQ0UsRUFBRSxFQUFFLEtBQUs7WUFDVCxNQUFNLEVBQUUsb01BQW9NO1NBQzdNO1FBQ0Q7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSxpRkFBaUY7U0FDMUY7UUFDRDtZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLGlFQUFpRTtTQUMxRTtLQUNGLENBQUMsQ0FBQztJQUVILDRDQUE0QztJQUM1QyxNQUFNLGNBQWMsR0FBRyxzQ0FBc0MsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBa0IsQ0FBQztJQUMxRywyQkFBbUIsQ0FBQyxjQUFjLEVBQUU7UUFDbEM7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSxrR0FBa0c7U0FDM0c7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDO0FBOUJELG9HQThCQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8qXG4gKiAgVGhlIGZ1bmN0aW9ucyBmb3VuZCBoZXJlIGluIHRoZSBjb3JlIGxpYnJhcnkgYXJlIGZvciBpbnRlcm5hbCB1c2UgYW5kIGNhbiBiZSBjaGFuZ2VkXG4gKiAgb3IgcmVtb3ZlZCBvdXRzaWRlIG9mIGEgbWFqb3IgcmVsZWFzZS4gV2UgcmVjb21tZW5kIGFnYWluc3QgY2FsbGluZyB0aGVtIGRpcmVjdGx5IGZyb20gY2xpZW50IGNvZGUuXG4gKi9cblxuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBEZWZhdWx0UzNQcm9wcyB9IGZyb20gJy4vczMtYnVja2V0LWRlZmF1bHRzJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMsIGFkZENmblN1cHByZXNzUnVsZXMgfSBmcm9tICcuL3V0aWxzJztcbmltcG9ydCB7IFN0b3JhZ2VDbGFzcyB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliJztcbi8vIE5vdGU6IFRvIGVuc3VyZSBDREt2MiBjb21wYXRpYmlsaXR5LCBrZWVwIHRoZSBpbXBvcnQgc3RhdGVtZW50IGZvciBDb25zdHJ1Y3Qgc2VwYXJhdGVcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkUzNCdWNrZXRQcm9wcyB7XG4gIC8qKlxuICAgKiBVc2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgUzMgQnVja2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldFByb3BzPzogczMuQnVja2V0UHJvcHM7XG4gIC8qKlxuICAgKiBVc2VyIHByb3ZpZGVkIHByb3BzIHRvIG92ZXJyaWRlIHRoZSBkZWZhdWx0IHByb3BzIGZvciB0aGUgUzMgTG9nZ2luZyBCdWNrZXQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRGVmYXVsdCBwcm9wcyBhcmUgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgbG9nZ2luZ0J1Y2tldFByb3BzPzogczMuQnVja2V0UHJvcHM7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIHR1cm4gb24gQWNjZXNzIExvZ3MgZm9yIFMzLiBVc2VzIGFuIFMzIGJ1Y2tldCB3aXRoIGFzc29jaWF0ZWQgc3RvcmFnZSBjb3N0cy5cbiAgICogRW5hYmxpbmcgQWNjZXNzIExvZ2dpbmcgaXMgYSBiZXN0IHByYWN0aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGxvZ1MzQWNjZXNzTG9ncz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQGludGVybmFsIFRoaXMgaXMgYW4gaW50ZXJuYWwgY29yZSBmdW5jdGlvbiBhbmQgc2hvdWxkIG5vdCBiZSBjYWxsZWQgZGlyZWN0bHkgYnkgU29sdXRpb25zIENvbnN0cnVjdHMgY2xpZW50cy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUxvZ2dpbmdCdWNrZXQoc2NvcGU6IENvbnN0cnVjdCxcbiAgYnVja2V0SWQ6IHN0cmluZyxcbiAgbG9nZ2luZ0J1Y2tldFByb3BzOiBzMy5CdWNrZXRQcm9wcyk6IHMzLkJ1Y2tldCB7XG5cbiAgLy8gQ3JlYXRlIHRoZSBMb2dnaW5nIEJ1Y2tldFxuICBjb25zdCBsb2dnaW5nQnVja2V0OiBzMy5CdWNrZXQgPSBuZXcgczMuQnVja2V0KHNjb3BlLCBidWNrZXRJZCwgbG9nZ2luZ0J1Y2tldFByb3BzKTtcblxuICAvLyBFeHRyYWN0IHRoZSBDZm5CdWNrZXQgZnJvbSB0aGUgbG9nZ2luZ0J1Y2tldFxuICBjb25zdCBsb2dnaW5nQnVja2V0UmVzb3VyY2UgPSBsb2dnaW5nQnVja2V0Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIHMzLkNmbkJ1Y2tldDtcblxuICAvLyBPdmVycmlkZSBhY2Nlc3NDb250cm9sIGNvbmZpZ3VyYXRpb24gYW5kIGFkZCBtZXRhZGF0YSBmb3IgdGhlIGxvZ2dpbmcgYnVja2V0XG4gIGxvZ2dpbmdCdWNrZXRSZXNvdXJjZS5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdBY2Nlc3NDb250cm9sJywgJ0xvZ0RlbGl2ZXJ5V3JpdGUnKTtcblxuICAvLyBSZW1vdmUgdGhlIGRlZmF1bHQgTGlmZWN5Y2xlQ29uZmlndXJhdGlvbiBmb3IgdGhlIExvZ2dpbmcgQnVja2V0XG4gIGxvZ2dpbmdCdWNrZXRSZXNvdXJjZS5hZGRQcm9wZXJ0eURlbGV0aW9uT3ZlcnJpZGUoJ0xpZmVjeWNsZUNvbmZpZ3VyYXRpb24uUnVsZXMnKTtcblxuICBsZXQgX3JlYXNvbiA9IFwiVGhpcyBTMyBidWNrZXQgaXMgdXNlZCBhcyB0aGUgYWNjZXNzIGxvZ2dpbmcgYnVja2V0IGZvciBhbm90aGVyIGJ1Y2tldFwiO1xuXG4gIGlmIChidWNrZXRJZCA9PT0gJ0Nsb3VkZnJvbnRMb2dnaW5nQnVja2V0Jykge1xuICAgIF9yZWFzb24gPSBcIlRoaXMgUzMgYnVja2V0IGlzIHVzZWQgYXMgdGhlIGFjY2VzcyBsb2dnaW5nIGJ1Y2tldCBmb3IgQ2xvdWRGcm9udCBEaXN0cmlidXRpb25cIjtcbiAgfVxuXG4gIGFkZENmblN1cHByZXNzUnVsZXMobG9nZ2luZ0J1Y2tldFJlc291cmNlLCBbXG4gICAge1xuICAgICAgaWQ6ICdXMzUnLFxuICAgICAgcmVhc29uOiBfcmVhc29uXG4gICAgfVxuICBdKTtcblxuICByZXR1cm4gbG9nZ2luZ0J1Y2tldDtcbn1cblxuLyoqXG4gKiBAaW50ZXJuYWwgVGhpcyBpcyBhbiBpbnRlcm5hbCBjb3JlIGZ1bmN0aW9uIGFuZCBzaG91bGQgbm90IGJlIGNhbGxlZCBkaXJlY3RseSBieSBTb2x1dGlvbnMgQ29uc3RydWN0cyBjbGllbnRzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY3JlYXRlQWxiTG9nZ2luZ0J1Y2tldChzY29wZTogQ29uc3RydWN0LFxuICBidWNrZXRJZDogc3RyaW5nLFxuICBsb2dnaW5nQnVja2V0UHJvcHM6IHMzLkJ1Y2tldFByb3BzKTogczMuQnVja2V0IHtcblxuICAvLyBDcmVhdGUgdGhlIExvZ2dpbmcgQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXQ6IHMzLkJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc2NvcGUsIGJ1Y2tldElkLCBsb2dnaW5nQnVja2V0UHJvcHMpO1xuXG4gIC8vIEV4dHJhY3QgdGhlIENmbkJ1Y2tldCBmcm9tIHRoZSBsb2dnaW5nQnVja2V0XG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXRSZXNvdXJjZSA9IGxvZ2dpbmdCdWNrZXQubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgczMuQ2ZuQnVja2V0O1xuXG4gIGFkZENmblN1cHByZXNzUnVsZXMobG9nZ2luZ0J1Y2tldFJlc291cmNlLCBbXG4gICAge1xuICAgICAgaWQ6ICdXMzUnLFxuICAgICAgcmVhc29uOiBcIlRoaXMgaXMgYSBsb2cgYnVja2V0IGZvciBhbiBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyXCJcbiAgICB9XG4gIF0pO1xuXG4gIHJldHVybiBsb2dnaW5nQnVja2V0O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkUzNCdWNrZXRSZXNwb25zZSB7XG4gIHJlYWRvbmx5IGJ1Y2tldDogczMuQnVja2V0LFxuICByZWFkb25seSBsb2dnaW5nQnVja2V0PzogczMuQnVja2V0XG59XG5cbi8qKlxuICogQGludGVybmFsIFRoaXMgaXMgYW4gaW50ZXJuYWwgY29yZSBmdW5jdGlvbiBhbmQgc2hvdWxkIG5vdCBiZSBjYWxsZWQgZGlyZWN0bHkgYnkgU29sdXRpb25zIENvbnN0cnVjdHMgY2xpZW50cy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGJ1aWxkUzNCdWNrZXQoc2NvcGU6IENvbnN0cnVjdCxcbiAgcHJvcHM6IEJ1aWxkUzNCdWNrZXRQcm9wcyxcbiAgYnVja2V0SWQ/OiBzdHJpbmcpOiBCdWlsZFMzQnVja2V0UmVzcG9uc2Uge1xuXG4gIC8qKiBEZWZhdWx0IExpZmUgQ3ljbGUgcG9saWN5IHRvIHRyYW5zaXRpb24gb2xkZXIgdmVyc2lvbnMgdG8gR2xhY2llciBhZnRlciA5MCBkYXlzICovXG4gIGNvbnN0IGxpZmVjeWNsZVJ1bGVzOiBzMy5MaWZlY3ljbGVSdWxlW10gPSBbe1xuICAgIG5vbmN1cnJlbnRWZXJzaW9uVHJhbnNpdGlvbnM6IFt7XG4gICAgICBzdG9yYWdlQ2xhc3M6IFN0b3JhZ2VDbGFzcy5HTEFDSUVSLFxuICAgICAgdHJhbnNpdGlvbkFmdGVyOiBEdXJhdGlvbi5kYXlzKDkwKVxuICAgIH1dXG4gIH1dO1xuXG4gIC8vIENyZWF0ZSB0aGUgQXBwbGljYXRpb24gQnVja2V0XG4gIGxldCBjdXN0b21CdWNrZXRQcm9wczogczMuQnVja2V0UHJvcHM7XG4gIGxldCBsb2dnaW5nQnVja2V0O1xuICBjb25zdCBfYnVja2V0SWQgPSBidWNrZXRJZCA/IGJ1Y2tldElkICsgJ1MzQnVja2V0JyA6ICdTM0J1Y2tldCc7XG4gIGNvbnN0IF9sb2dnaW5nQnVja2V0SWQgPSBidWNrZXRJZCA/IGJ1Y2tldElkICsgJ1MzTG9nZ2luZ0J1Y2tldCcgOiAnUzNMb2dnaW5nQnVja2V0JztcblxuICAvLyBJZiBsb2dnaW5nIFMzIGFjY2VzcyBsb2dzIGlzIGVuYWJsZWQvdW5kZWZpbmVkIGFuZCBhbiBleGlzdGluZyBidWNrZXQgb2JqZWN0IGlzIG5vdCBwcm92aWRlZFxuICBpZiAocHJvcHMubG9nUzNBY2Nlc3NMb2dzICE9PSBmYWxzZSAmJiAhKHByb3BzLmJ1Y2tldFByb3BzPy5zZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0KSkge1xuICAgIC8vIENyZWF0ZSB0aGUgTG9nZ2luZyBCdWNrZXRcbiAgICBsZXQgbG9nZ2luZ0J1Y2tldFByb3BzID0gRGVmYXVsdFMzUHJvcHMoKTtcblxuICAgIGlmIChwcm9wcy5sb2dnaW5nQnVja2V0UHJvcHMpIHtcbiAgICAgIC8vIFVzZXIgcHJvdmlkZWQgbG9nZ2luZyBidWNrZXQgcHJvcHNcbiAgICAgIGxvZ2dpbmdCdWNrZXRQcm9wcyA9IG92ZXJyaWRlUHJvcHMobG9nZ2luZ0J1Y2tldFByb3BzLCBwcm9wcy5sb2dnaW5nQnVja2V0UHJvcHMpO1xuICAgIH0gZWxzZSBpZiAocHJvcHMuYnVja2V0UHJvcHM/LnJlbW92YWxQb2xpY3kpIHtcbiAgICAgIC8vIElmIHRoZSBjbGllbnQgZXhwbGljaXRseSBzcGVjaWZpZWQgYSByZW1vdmFsIHBvbGljeSBmb3IgdGhlIG1haW4gYnVja2V0LFxuICAgICAgLy8gdGhlbiByZXBsaWNhdGUgdGhhdCBwb2xpY3kgb24gdGhlIGxvZ2dpbmcgYnVja2V0XG4gICAgICBsb2dnaW5nQnVja2V0UHJvcHMgPSBvdmVycmlkZVByb3BzKGxvZ2dpbmdCdWNrZXRQcm9wcywgeyByZW1vdmFsUG9saWN5OiBwcm9wcy5idWNrZXRQcm9wcy5yZW1vdmFsUG9saWN5IH0pO1xuICAgIH1cblxuICAgIGxvZ2dpbmdCdWNrZXQgPSBjcmVhdGVMb2dnaW5nQnVja2V0KHNjb3BlLCBfbG9nZ2luZ0J1Y2tldElkLCBsb2dnaW5nQnVja2V0UHJvcHMpO1xuICB9XG4gIC8vIEF0dGFjaCB0aGUgRGVmYXVsdCBMaWZlIEN5Y2xlIHBvbGljeSBPTkxZIElGIHRoZSB2ZXJzaW9uaW5nIGlzIEVOQUJMRURcbiAgaWYgKHByb3BzLmJ1Y2tldFByb3BzPy52ZXJzaW9uZWQgPT09IHVuZGVmaW5lZCB8fCBwcm9wcy5idWNrZXRQcm9wcy52ZXJzaW9uZWQpIHtcbiAgICBjdXN0b21CdWNrZXRQcm9wcyA9IERlZmF1bHRTM1Byb3BzKGxvZ2dpbmdCdWNrZXQsIGxpZmVjeWNsZVJ1bGVzKTtcbiAgfSBlbHNlIHtcbiAgICBjdXN0b21CdWNrZXRQcm9wcyA9IERlZmF1bHRTM1Byb3BzKGxvZ2dpbmdCdWNrZXQpO1xuICB9XG5cbiAgY3VzdG9tQnVja2V0UHJvcHMgPSBwcm9wcy5idWNrZXRQcm9wcyA/IG92ZXJyaWRlUHJvcHMoY3VzdG9tQnVja2V0UHJvcHMsIHByb3BzLmJ1Y2tldFByb3BzKSA6IGN1c3RvbUJ1Y2tldFByb3BzO1xuXG4gIGNvbnN0IHMzQnVja2V0OiBzMy5CdWNrZXQgPSBuZXcgczMuQnVja2V0KHNjb3BlLCBfYnVja2V0SWQsIGN1c3RvbUJ1Y2tldFByb3BzICk7XG5cbiAgcmV0dXJuIHsgYnVja2V0OiBzM0J1Y2tldCwgbG9nZ2luZ0J1Y2tldCB9O1xufVxuXG4vKipcbiAqIEBpbnRlcm5hbCBUaGlzIGlzIGFuIGludGVybmFsIGNvcmUgZnVuY3Rpb24gYW5kIHNob3VsZCBub3QgYmUgY2FsbGVkIGRpcmVjdGx5IGJ5IFNvbHV0aW9ucyBDb25zdHJ1Y3RzIGNsaWVudHMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBhZGRDZm5OYWdTM0J1Y2tldE5vdGlmaWNhdGlvblJ1bGVzVG9TdXBwcmVzcyhzdGFja1Jvb3Q6IGNkay5TdGFjaywgbG9naWNhbElkOiBzdHJpbmcpIHtcbiAgY29uc3Qgbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlciA9IHN0YWNrUm9vdC5ub2RlLnRyeUZpbmRDaGlsZChsb2dpY2FsSWQpIGFzIGxhbWJkYS5GdW5jdGlvbjtcbiAgY29uc3Qgbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlclJvbGVSb2xlID0gbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlci5ub2RlLmZpbmRDaGlsZCgnUm9sZScpIGFzIGlhbS5Sb2xlO1xuICBjb25zdCBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyUm9sZVBvbGljeSA9IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXJSb2xlUm9sZS5ub2RlLmZpbmRDaGlsZCgnRGVmYXVsdFBvbGljeScpIGFzIGlhbS5Qb2xpY3k7XG5cbiAgLy8gRXh0cmFjdCB0aGUgQ2ZuRnVuY3Rpb24gZnJvbSB0aGUgRnVuY3Rpb25cbiAgY29uc3QgZm5SZXNvdXJjZSA9IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXIubm9kZS5maW5kQ2hpbGQoJ1Jlc291cmNlJykgYXMgbGFtYmRhLkNmbkZ1bmN0aW9uO1xuICBhZGRDZm5TdXBwcmVzc1J1bGVzKGZuUmVzb3VyY2UsIFtcbiAgICB7XG4gICAgICBpZDogJ1c1OCcsXG4gICAgICByZWFzb246IGBMYW1iZGEgZnVuY3Rpb25zIGhhcyB0aGUgcmVxdWlyZWQgcGVybWlzc2lvbiB0byB3cml0ZSBDbG91ZFdhdGNoIExvZ3MuIEl0IHVzZXMgY3VzdG9tIHBvbGljeSBpbnN0ZWFkIG9mIGFybjphd3M6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUgd2l0aCB0aWdodGVyIHBlcm1pc3Npb25zLmBcbiAgICB9LFxuICAgIHtcbiAgICAgIGlkOiAnVzg5JyxcbiAgICAgIHJlYXNvbjogYFRoaXMgaXMgbm90IGEgcnVsZSBmb3IgdGhlIGdlbmVyYWwgY2FzZSwganVzdCBmb3Igc3BlY2lmaWMgdXNlIGNhc2VzL2luZHVzdHJpZXNgXG4gICAgfSxcbiAgICB7XG4gICAgICBpZDogJ1c5MicsXG4gICAgICByZWFzb246IGBJbXBvc3NpYmxlIGZvciB1cyB0byBkZWZpbmUgdGhlIGNvcnJlY3QgY29uY3VycmVuY3kgZm9yIGNsaWVudHNgXG4gICAgfVxuICBdKTtcblxuICAvLyBFeHRyYWN0IHRoZSBDZm5Qb2xpY3kgZnJvbSB0aGUgaWFtLlBvbGljeVxuICBjb25zdCBwb2xpY3lSZXNvdXJjZSA9IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXJSb2xlUG9saWN5Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGlhbS5DZm5Qb2xpY3k7XG4gIGFkZENmblN1cHByZXNzUnVsZXMocG9saWN5UmVzb3VyY2UsIFtcbiAgICB7XG4gICAgICBpZDogJ1cxMicsXG4gICAgICByZWFzb246IGBCdWNrZXQgcmVzb3VyY2UgaXMgJyonIGR1ZSB0byBjaXJjdWxhciBkZXBlbmRlbmN5IHdpdGggYnVja2V0IGFuZCByb2xlIGNyZWF0aW9uIGF0IHRoZSBzYW1lIHRpbWVgXG4gICAgfVxuICBdKTtcbn1cbiJdfQ==