"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const sagemaker = require("aws-cdk-lib/aws-sagemaker");
const ec2 = require("aws-cdk-lib/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(role, props) {
    // Grant permissions to NoteBookInstance for creating and training the model
    role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:sagemaker:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = props.endpointConfigProps
            ?.productionVariants[0].acceleratorType;
        if (acceleratorType !== undefined) {
            role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    role.addToPolicy(new iam.PolicyStatement({
        resources: [role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    role.addToPolicy(new iam.PolicyStatement({
        resources: [role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefaultPolicy = role.node.tryFindChild('DefaultPolicy')?.node.findChild('Resource');
    utils_1.addCfnSuppressRules(roleDefaultPolicy, [
        {
            id: 'W12',
            reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
        },
        {
            id: 'W76',
            reason: 'Complex role becuase Sagemaker needs permissions to access several services',
        }
    ]);
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildSagemakerNotebook(scope, props) {
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((props.sagemakerNotebookProps?.subnetId && props.sagemakerNotebookProps?.securityGroupIds === undefined) ||
            (props.sagemakerNotebookProps?.subnetId === undefined && props.sagemakerNotebookProps?.securityGroupIds)) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (props.sagemakerNotebookProps?.kmsKeyId === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (props.sagemakerNotebookProps?.subnetId === undefined &&
                props.sagemakerNotebookProps?.securityGroupIds === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, props.sagemakerNotebookProps?.subnetId, props.sagemakerNotebookProps?.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        sagemakerNotebookProps = utils_1.consolidateProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return { notebook: sagemakerInstance, vpc: vpcInstance, securityGroup };
        }
        else {
            return { notebook: sagemakerInstance };
        }
    }
    else {
        // Return existing notebook object
        return { notebook: props.existingNotebookObj };
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            const deploySagemakerEndpointResponse = deploySagemakerEndpoint(scope, props);
            return { ...deploySagemakerEndpointResponse };
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return { endpoint: props.existingSagemakerEndpointObj };
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependency(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependency(endpointConfig);
        return { endpoint, endpointConfig, model };
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        utils_1.addCfnSuppressRules(cfnSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig), modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Overwrite default EndpointConfig properties
    finalEndpointConfigProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId), endpointConfigProps);
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Overwrite default Endpoint properties
    finalEndpointProps = utils_1.consolidateProps(sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName), endpointProps);
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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