"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
// Imports
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(aws_cdk_lib_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": aws_cdk_lib_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        const snsTopicProps = utils_1.consolidateProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        if ((props.topicProps?.masterKey || props.encryptionKey || props.encryptionKeyProps) && props.enableEncryptionWithCustomerManagedKey === true) {
            utils_1.printWarning("Ignoring enableEncryptionWithCustomerManagedKey because one of topicProps.masterKey, encryptionKey, or encryptionKeyProps was already specified");
        }
        // Set encryption properties
        if (props.topicProps?.masterKey) {
            snsTopicProps.masterKey = props.topicProps?.masterKey;
        }
        else if (props.encryptionKey) {
            snsTopicProps.masterKey = props.encryptionKey;
        }
        else if (props.encryptionKeyProps || props.enableEncryptionWithCustomerManagedKey === true) {
            snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
        }
        else {
            snsTopicProps.masterKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return { topic, key: snsTopicProps.masterKey };
    }
    else {
        return { topic: props.existingTopicObj, key: props.existingTopicEncryptionKey };
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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