/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as sqs from 'aws-cdk-lib/aws-sqs';
import * as kms from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
export interface BuildQueueProps {
    /**
     * Existing instance of SQS queue object, providing both this and queueProps will cause an error.
     *
     * @default - None.
     */
    readonly existingQueueObj?: sqs.Queue;
    /**
     * Optional user provided props to override the default props for the primary queue.
     *
     * @default - Default props are used.
     */
    readonly queueProps?: sqs.QueueProps;
    /**
     * Optional dead letter queue to pass bad requests to after the max receive count is reached.
     *
     * @default - Default props are used.
     */
    readonly deadLetterQueue?: sqs.DeadLetterQueue;
    /**
     * If no key is provided, this flag determines whether the queue is encrypted with a new CMK or an AWS managed key.
     * This flag is ignored if any of the following are defined: queueProps.encryptionMasterKey, encryptionKey or encryptionKeyProps.
     *
     * @default - False if queueProps.encryptionMasterKey, encryptionKey, and encryptionKeyProps are all undefined.
     */
    readonly enableEncryptionWithCustomerManagedKey?: boolean;
    /**
     * An optional, imported encryption key to encrypt the SQS Queue with.
     *
     * @default - None
     */
    readonly encryptionKey?: kms.Key;
    /**
     * Optional user provided properties to override the default properties for the KMS encryption key used to encrypt the SQS Queue with.
     *
     * @default - None
     */
    readonly encryptionKeyProps?: kms.KeyProps;
}
export interface BuildQueueResponse {
    readonly queue: sqs.Queue;
    readonly key?: kms.IKey;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
export declare function buildQueue(scope: Construct, id: string, props: BuildQueueProps): BuildQueueResponse;
export interface BuildDeadLetterQueueProps {
    /**
     * Existing instance of SQS queue object, providing both this and queueProps will cause an error.
     *
     * @default - None.
     */
    readonly existingQueueObj?: sqs.Queue;
    /**
     * Whether to deploy a secondary queue to be used as a dead letter queue.
     *
     * @default - required field.
     */
    readonly deployDeadLetterQueue?: boolean;
    /**
     * Optional user provided properties for the dead letter queue
     *
     * @default - Default props are used
     */
    readonly deadLetterQueueProps?: sqs.QueueProps;
    /**
     * The number of times a message can be unsuccessfully dequeued before being moved to the dead letter queue.
     *
     * @default - Default props are used
     */
    readonly maxReceiveCount?: number;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
export declare function buildDeadLetterQueue(scope: Construct, props: BuildDeadLetterQueueProps): sqs.DeadLetterQueue | undefined;
