"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
// Imports
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kms_helper_1 = require("./kms-helper");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildQueue(scope, id, props) {
    if ((props.queueProps?.encryptionMasterKey || props.encryptionKey || props.encryptionKeyProps)
        && props.enableEncryptionWithCustomerManagedKey === true) {
        utils_1.printWarning(`Ignoring enableEncryptionWithCustomerManagedKey because one of
     queueProps.encryptionMasterKey, encryptionKey, or encryptionKeyProps was already specified`);
    }
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            const checkFifo = props.queueProps.fifo ? true : undefined;
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), { ...props.queueProps, fifo: checkFifo });
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties.
        // Note that defaults.DefaultQueueProps sets encryption to Server-side KMS encryption with a KMS key managed by SQS.
        if (props.queueProps?.encryptionMasterKey) {
            queueProps.encryptionMasterKey = props.queueProps?.encryptionMasterKey;
        }
        else if (props.encryptionKey) {
            queueProps.encryptionMasterKey = props.encryptionKey;
        }
        else if (props.encryptionKeyProps || props.enableEncryptionWithCustomerManagedKey === true) {
            queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return { queue, key: queue.encryptionMasterKey };
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return { queue: props.existingQueueObj };
    }
}
exports.buildQueue = buildQueue;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const buildQueueResponse = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: buildQueueResponse.queue
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(aws_cdk_lib_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply queue policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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