"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("aws-cdk-lib");
const smDefaults = require("./step-function-defaults");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    let logGroup;
    let consolidatedStateMachineProps;
    // If they sent a logGroup in stateMachineProps
    if (stateMachineProps.logs?.destination) {
        logGroup = stateMachineProps.logs?.destination;
        consolidatedStateMachineProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!consolidatedLogGroupProps?.logGroupName) {
            const logGroupPrefix = '/aws/vendedlogs/states/constructs/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength, true);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        consolidatedStateMachineProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const newStateMachine = new sfn.StateMachine(scope, 'StateMachine', consolidatedStateMachineProps);
    // If the client did not pass a role we got the default role and will trim the privileges
    if (!stateMachineProps.role) {
        const role = newStateMachine.node.findChild('Role');
        const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
        // Reduce the scope of actions for the existing DefaultPolicy
        cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
            "logs:CreateLogDelivery",
            'logs:GetLogDelivery',
            'logs:UpdateLogDelivery',
            'logs:DeleteLogDelivery',
            'logs:ListLogDeliveries'
        ]);
        // Override Cfn Nag warning W12: IAM policy should not allow * resource
        utils_1.addCfnSuppressRules(cfnDefaultPolicy, [
            {
                id: 'W12',
                reason: `The 'LogDelivery' actions do not support resource-level authorizations`
            }
        ]);
        // Add a new policy with logging permissions for the given cloudwatch log group
        newStateMachine.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:PutResourcePolicy',
                'logs:DescribeResourcePolicies',
                'logs:DescribeLogGroups'
            ],
            resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
        }));
    }
    return { stateMachine: newStateMachine, logGroup };
}
exports.buildStateMachine = buildStateMachine;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted({
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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