"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateName = exports.consolidateProps = exports.addCfnSuppressRules = exports.generateResourceName = exports.printWarning = exports.overrideProps = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const deepmerge = require("deepmerge");
const override_warning_service_1 = require("./override-warning-service");
const log = require("npmlog");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
function isObject(val) {
    return val != null && typeof val === 'object'
        && Object.prototype.toString.call(val) === '[object Object]';
}
function isPlainObject(o) {
    if (Array.isArray(o) === true) {
        return true;
    }
    if (isObject(o) === false) {
        return false;
    }
    // If has modified constructor
    const ctor = o.constructor;
    if (typeof ctor !== 'function') {
        return false;
    }
    // If has modified prototype
    const prot = ctor.prototype;
    if (isObject(prot) === false) {
        return false;
    }
    // If constructor does not have an Object-specific method
    if (prot.hasOwnProperty('isPrototypeOf') === false) {
        return false;
    }
    // Most likely a plain Object
    return true;
}
function combineMerge(target, source) {
    return target.concat(source);
}
function overwriteMerge(target, source) {
    target = source;
    return target;
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function overrideProps(DefaultProps, userProps, concatArray = false) {
    // Notify the user via console output if defaults are overridden
    const overrideWarningsEnabled = (process.env.overrideWarningsEnabled !== 'false');
    if (overrideWarningsEnabled) {
        override_warning_service_1.flagOverriddenDefaults(DefaultProps, userProps);
    }
    // Override the sensible defaults with user provided props
    if (concatArray) {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: combineMerge,
            isMergeableObject: isPlainObject
        });
    }
    else {
        return deepmerge(DefaultProps, userProps, {
            arrayMerge: overwriteMerge,
            isMergeableObject: isPlainObject
        });
    }
}
exports.overrideProps = overrideProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function printWarning(message) {
    // Style the log output
    log.prefixStyle.bold = true;
    log.prefixStyle.fg = 'red';
    log.enableColor();
    log.warn('AWS_SOLUTIONS_CONSTRUCTS_WARNING: ', message);
}
exports.printWarning = printWarning;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * @summary Creates a resource name in the style of the CDK (string+hash)
 * @param {string[]} parts - the various string components of the name (eg - stackName, solutions construct ID, L2 construct ID)
 * @param {number} maxLength - the longest string that can be returned
 * @returns {string} - a string with concatenated parts (truncated if neccessary) + a hash of the full concatenated parts
 *
 * This is based upon this discussion - https://github.com/aws/aws-cdk/issues/1424
 */
function generateResourceName(parts, maxLength, randomize = false) {
    const hashLength = 12;
    const randomizor = randomize ? (new Date()).getTime().toString() : "";
    const maxPartLength = Math.floor((maxLength - hashLength - randomizor.length) / parts.length);
    const sha256 = crypto.createHash("sha256");
    let finalName = '';
    parts.forEach((part) => {
        sha256.update(part);
        finalName += removeNonAlphanumeric(part.slice(0, maxPartLength));
    });
    const hash = sha256.digest("hex").slice(0, hashLength);
    finalName += hash;
    finalName += randomizor;
    return finalName.toLowerCase();
}
exports.generateResourceName = generateResourceName;
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Adds CFN NAG suppress rules to the CDK resource.
 * @param resource The CDK resource
 * @param rules The CFN NAG suppress rules
 */
function addCfnSuppressRules(resource, rules) {
    if (resource instanceof cdk.Resource) {
        resource = resource.node.defaultChild;
    }
    if (resource.cfnOptions.metadata?.cfn_nag?.rules_to_suppress) {
        resource.cfnOptions.metadata?.cfn_nag.rules_to_suppress.push(...rules);
    }
    else {
        resource.addMetadata('cfn_nag', {
            rules_to_suppress: rules
        });
    }
}
exports.addCfnSuppressRules = addCfnSuppressRules;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Creates the props to be used to instantiate a CDK L2 construct within a Solutions Construct
 *
 * @param defaultProps The default props to be used by the construct
 * @param clientProps Optional properties passed in from the client in the props object
 * @param constructProps Optional properties required by the construct for the construct to work (override any other values)
 * @returns The properties to use - all values prioritized:
 *  1) constructProps value
 *  2) clientProps value
 *  3) defaultProps value
 */
function consolidateProps(defaultProps, clientProps, constructProps, concatArray = false) {
    let result = defaultProps;
    if (clientProps) {
        result = overrideProps(result, clientProps, concatArray);
    }
    if (constructProps) {
        result = overrideProps(result, constructProps, concatArray);
    }
    return result;
}
exports.consolidateProps = consolidateProps;
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 *
 * Generates a name unique to this location in this stack with this stackname. Truncates to under 64 characters if needed.
 * (will allow 2 copies of the stack with different stack names, but will collide if both stacks have the same name)
 *
 * @param scope the construct within to create the name
 * @param resourceId an id for the construct about to be created under scope (empty string if name is for scoep)
 * @returns a unique name
 *
 * Note: This appears to overlap with GenerateResourceName above (I wrote it before noticing that
 * function). As this offloads the logic to the CDK, I'm leaving this here but someone may want to
 * blend these routines in the future.
 */
function generateName(scope, resourceId = "") {
    const name = resourceId + cdk.Names.uniqueId(scope);
    if (name.length > 64) {
        return name.substring(0, 32) + name.substring(name.length - 32);
    }
    return name;
}
exports.generateName = generateName;
//# sourceMappingURL=data:application/json;base64,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