"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.retrievePrivateSubnetIds = exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
/*
 *  The functions found here in the core library are for internal use and can be changed
 *  or removed outside of a major release. We recommend against calling them directly from client code.
 */
const ec2 = require("aws-cdk-lib/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function buildVpc(scope, props) {
    if (props?.existingVpc) {
        return props?.existingVpc;
    }
    let cumulativeProps = props?.defaultVpcProps;
    cumulativeProps = utils_1.consolidateProps(cumulativeProps, props?.userVpcProps, props?.constructVpcProps);
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    SuppressMapPublicIpWarnings(vpc);
    SuppressEncryptedLogWarnings(flowLog);
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEP_FUNCTIONS"] = "STEP_FUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
    ServiceEndpointTypes["ECR_API"] = "ECR_API";
    ServiceEndpointTypes["ECR_DKR"] = "ECR_DKR";
    ServiceEndpointTypes["EVENTS"] = "CLOUDWATCH_EVENTS";
    ServiceEndpointTypes["KINESIS_FIREHOSE"] = "KINESIS_FIREHOSE";
    ServiceEndpointTypes["KINESIS_STREAMS"] = "KINESIS_STREAMS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.STEP_FUNCTIONS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
    {
        endpointName: ServiceEndpointTypes.ECR_API,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR
    },
    {
        endpointName: ServiceEndpointTypes.ECR_DKR,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
    },
    {
        endpointName: ServiceEndpointTypes.EVENTS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS
    },
    {
        endpointName: ServiceEndpointTypes.KINESIS_FIREHOSE,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE
    },
    {
        endpointName: ServiceEndpointTypes.KINESIS_STREAMS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.KINESIS_STREAMS
    }
];
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (CheckIfEndpointAlreadyExists(vpc, interfaceTag)) {
        return;
    }
    const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
    if (!service) {
        throw new Error("Unsupported Service sent to AddServiceEndpoint");
    }
    if (service.endpointType === EndpointTypes.GATEWAY) {
        AddGatewayEndpoint(vpc, service, interfaceTag);
    }
    if (service.endpointType === EndpointTypes.INTERFACE) {
        AddInterfaceEndpoint(scope, vpc, service, interfaceTag);
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
function CheckIfEndpointAlreadyExists(vpc, interfaceTag) {
    return vpc.node.children.some((child) => child.node.id === interfaceTag);
}
function SuppressMapPublicIpWarnings(vpc) {
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        utils_1.addCfnSuppressRules(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
            }
        ]);
    });
}
function SuppressEncryptedLogWarnings(flowLog) {
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = flowLog.logGroup?.node.defaultChild;
    utils_1.addCfnSuppressRules(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
        }
    ]);
}
function AddInterfaceEndpoint(scope, vpc, service, interfaceTag) {
    const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, `${scope.node.id}-${service.endpointName}`, {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
    vpc.addInterfaceEndpoint(interfaceTag, {
        service: service.endpointInterfaceService,
        securityGroups: [endpointDefaultSecurityGroup],
    });
}
function AddGatewayEndpoint(vpc, service, interfaceTag) {
    vpc.addGatewayEndpoint(interfaceTag, {
        service: service.endpointGatewayService,
    });
}
/**
 * @internal This is an internal core function and should not be called directly by Solutions Constructs clients.
 */
function retrievePrivateSubnetIds(vpc) {
    let targetSubnetType;
    if (vpc.isolatedSubnets.length) {
        targetSubnetType = ec2.SubnetType.PRIVATE_ISOLATED;
    }
    else if (vpc.privateSubnets.length) {
        targetSubnetType = ec2.SubnetType.PRIVATE_WITH_EGRESS;
    }
    else {
        throw new Error('Error - No isolated or private subnets available in VPC');
    }
    const subnetSelector = {
        onePerAz: true,
        subnetType: targetSubnetType
    };
    return vpc.selectSubnets(subnetSelector).subnetIds;
}
exports.retrievePrivateSubnetIds = retrievePrivateSubnetIds;
//# sourceMappingURL=data:application/json;base64,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