"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
const s3_bucket_helper_1 = require("../lib/s3-bucket-helper");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const acm = require("aws-cdk-lib/aws-certificatemanager");
test('check bucket policy metadata', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "F16",
                        reason: "Public website bucket policy requires a wildcard principal"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('check bucket metadata', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket);
    expect(stack).toHaveResource('AWS::S3::Bucket', {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W35",
                        reason: "This S3 bucket is used as the access logging bucket for CloudFront Distribution"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('test cloudfront check bucket policy', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket);
    expect(stack).toHaveResourceLike("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "S3Bucket07682993",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                },
                {
                    Action: "s3:GetObject",
                    Effect: "Allow",
                    Principal: {
                        CanonicalUser: {
                            "Fn::GetAtt": [
                                "CloudFrontDistributionOrigin1S3Origin3D9CA0E9",
                                "S3CanonicalUserId"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('test cloudfront with no security headers ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, {}, false);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override cloudfront logging bucket ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const logBucket = new aws_s3_1.Bucket(stack, 'loggingbucket');
    const myprops = {
        enableLogging: true,
        logBucket
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "SetHttpSecurityHeadersEE936115",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "loggingbucket6D73BD53",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const props = {
        defaultBehavior: {
            origin: new origins.S3Origin(buildS3BucketResponse.bucket, { originPath: '/testPath' }),
            viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            allowedMethods: cloudfront.AllowedMethods.ALLOW_ALL,
            cachedMethods: cloudfront.CachedMethods.CACHE_GET_HEAD_OPTIONS
        },
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PUT",
                    "PATCH",
                    "POST",
                    "DELETE"
                ],
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "SetHttpSecurityHeadersEE936115",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    OriginPath: '/testPath',
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront with custom cloudfront function', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom cloudfront function
    const cloudfrontFunction = new cloudfront.Function(stack, "MyFunction", {
        code: cloudfront.FunctionCode.fromInline("exports.handler = (event, context, callback) => {}")
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, {
        defaultBehavior: {
            functionAssociations: [
                {
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                    function: cloudfrontFunction
                }
            ],
        }
    });
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                FunctionAssociations: [
                    {
                        EventType: "viewer-response",
                        FunctionARN: {
                            "Fn::GetAtt": [
                                "MyFunction3BAA72D1",
                                "FunctionARN"
                            ]
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test override cloudfront replace custom lambda@edge', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    // custom lambda@edg function
    const handler = new lambda.Function(stack, 'SomeHandler', {
        functionName: 'SomeHandler',
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const handlerVersion = new lambda.Version(stack, 'SomeHandlerVersion', {
        lambda: handler,
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, {
        defaultBehavior: {
            edgeLambdas: [
                {
                    eventType: aws_cloudfront_1.LambdaEdgeEventType.VIEWER_REQUEST,
                    includeBody: false,
                    functionVersion: handlerVersion,
                }
            ]
        }
    }, false);
    expect(stack).toHaveResource("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                CachePolicyId: "658327ea-f89d-4fab-a63d-7e88639e58f6",
                Compress: true,
                LambdaFunctionAssociations: [
                    {
                        EventType: "viewer-request",
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: "SomeHandlerVersionDA986E41"
                        }
                    }
                ],
                TargetOriginId: "CloudFrontDistributionOrigin176EC3A12",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                }
            },
            Origins: [
                {
                    DomainName: {
                        "Fn::GetAtt": [
                            "S3Bucket07682993",
                            "RegionalDomainName"
                        ]
                    },
                    Id: "CloudFrontDistributionOrigin176EC3A12",
                    S3OriginConfig: {
                        OriginAccessIdentity: {
                            "Fn::Join": [
                                "",
                                [
                                    "origin-access-identity/cloudfront/",
                                    {
                                        Ref: "CloudFrontDistributionOrigin1S3Origin3D9CA0E9"
                                    }
                                ]
                            ]
                        }
                    }
                }
            ]
        }
    });
});
test('test cloudfront override cloudfront custom domain names ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = s3_bucket_helper_1.buildS3Bucket(stack, {});
    const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012');
    const myprops = {
        domainNames: ['mydomains'],
        certificate
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForS3(stack, buildS3BucketResponse.bucket, myprops);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            Aliases: [
                "mydomains"
            ],
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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