"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test override for buildUserPool', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpoolProps = {
        userPoolName: 'test',
        signInAliases: { username: false, email: true, phone: true }
    };
    defaults.buildUserPool(stack, userpoolProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        UsernameAttributes: [
            "email",
            "phone_number"
        ],
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test"
    });
});
test('Test override for buildUserPoolClient', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclientProps = {
        userPoolClientName: 'test',
        userPool: userpool
    };
    defaults.buildUserPoolClient(stack, userpool, userpoolclientProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        UserPoolId: {
            Ref: "CognitoUserPool53E37E69"
        },
        ClientName: "test"
    });
});
test('Test override for buildIdentityPool', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    defaults.buildIdentityPool(stack, userpool, userpoolclient, {
        allowUnauthenticatedIdentities: true
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPool', {
        AllowUnauthenticatedIdentities: true,
        CognitoIdentityProviders: [
            {
                ClientId: {
                    Ref: "CognitoUserPoolClient5AB59AE4"
                },
                ProviderName: {
                    "Fn::GetAtt": [
                        "CognitoUserPool53E37E69",
                        "ProviderName"
                    ]
                },
                ServerSideTokenCheck: true
            }
        ]
    });
});
test('Test setupCognitoForSearchService', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    defaults.setupCognitoForSearchService(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain"
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPoolRoleAttachment', {
        IdentityPoolId: {
            Ref: "CognitoIdentityPool"
        },
        Roles: {
            authenticated: {
                "Fn::GetAtt": [
                    "CognitoAuthorizedRole14E74FE0",
                    "Arn"
                ]
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRoleWithWebIdentity",
                    Condition: {
                        "StringEquals": {
                            "cognito-identity.amazonaws.com:aud": {
                                Ref: "CognitoIdentityPool"
                            }
                        },
                        "ForAnyValue:StringLike": {
                            "cognito-identity.amazonaws.com:amr": "authenticated"
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Federated: "cognito-identity.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "es:ESHttp*",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":es:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":domain/test-domain/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "CognitoAccessPolicy"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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