"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const test_helper_1 = require("./test-helper");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
test("Test returning existing Cache", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const existingCache = test_helper_1.CreateTestCache(stack, 'test', testVpc);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    const obtainedCache = elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        existingCache,
        cacheSecurityGroupId: securityGroup.securityGroupId
    });
    expect(obtainedCache).toBe(existingCache);
});
test("Test create cache with no client props", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        vpc: testVpc,
        cacheSecurityGroupId: securityGroup.securityGroupId,
        cachePort: 11111,
    });
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        Port: 11111,
        AZMode: 'cross-az',
        Engine: 'memcached',
    });
});
test("Test create cache with client props", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const securityGroup = new ec2.SecurityGroup(stack, 'test-sg', {
        vpc: testVpc
    });
    elasticache_helper_1.obtainMemcachedCluster(stack, 'test-cache', {
        vpc: testVpc,
        cacheSecurityGroupId: securityGroup.securityGroupId,
        cachePort: 12321,
        cacheProps: {
            azMode: 'single-az',
            clusterName: 'test-name'
        }
    });
    expect(stack).toHaveResourceLike("AWS::ElastiCache::CacheCluster", {
        Port: 12321,
        AZMode: 'single-az',
        Engine: 'memcached',
        ClusterName: 'test-name'
    });
});
test("Test GetCachePort() with existing cache", () => {
    const stack = new cdk.Stack();
    const testVpc = test_helper_1.getTestVpc(stack, false);
    const existingCache = test_helper_1.CreateTestCache(stack, 'test', testVpc, 32123);
    const port = elasticache_helper_1.getCachePort(undefined, existingCache);
    // Since the port from the existing cache is a token,
    // we can't check it directly, but we can ensure
    // the default port was replaced
    expect(port).not.toEqual(elasticache_defaults_1.GetDefaultCachePort());
});
test("Test GetCachePort() with clientCacheProps", () => {
    const clientPort = 32123;
    const port = elasticache_helper_1.getCachePort({ port: clientPort });
    expect(port).toEqual(clientPort);
});
test("Test GetCachePort() with default port", () => {
    const port = elasticache_helper_1.getCachePort();
    expect(port).toEqual(elasticache_defaults_1.GetDefaultCachePort());
});
//# sourceMappingURL=data:application/json;base64,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