"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("..");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const __1 = require("..");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const ecr = require("aws-cdk-lib/aws-ecr");
require("@aws-cdk/assert/jest");
test('Test with all defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn);
    expect(createFargateServiceResponse.containerDefinition).toBeDefined();
    expect(createFargateServiceResponse.service).toBeDefined();
    expect(stack).toHaveResource("AWS::ECS::Service", {
        Cluster: {
            Ref: "testclusterDF8B0D19"
        },
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        DesiredCount: 2,
        EnableECSManagedTags: false,
        LaunchType: "FARGATE",
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "testsg872EB48A",
                            "GroupId"
                        ]
                    }
                ],
                Subnets: [
                    {
                        Ref: "VpcPrivateSubnet1Subnet536B997A"
                    },
                    {
                        Ref: "VpcPrivateSubnet2Subnet3788AAA1"
                    }
                ]
            }
        },
        PlatformVersion: "LATEST",
        TaskDefinition: {
            Ref: "testtaskdefF924AD58"
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
            }
        ],
    });
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroup", {
        GroupDescription: 'Construct created security group'
    });
    expect(stack).toCountResources("AWS::EC2::VPCEndpoint", 3);
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
test('Test with all defaults in isolated VPC', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn);
    expect(stack).toHaveResource("AWS::ECS::Service", {
        Cluster: {
            Ref: "testclusterDF8B0D19"
        },
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        DesiredCount: 2,
        EnableECSManagedTags: false,
        LaunchType: "FARGATE",
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "testsg872EB48A",
                            "GroupId"
                        ]
                    }
                ],
                Subnets: [
                    {
                        Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                    },
                    {
                        Ref: "VpcisolatedSubnet2Subnet39217055"
                    }
                ]
            }
        },
        PlatformVersion: "LATEST",
        TaskDefinition: {
            Ref: "testtaskdefF924AD58"
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
            }
        ],
    });
    expect(stack).toCountResources("AWS::EC2::VPCEndpoint", 3);
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
test('Test with custom task definition', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, undefined, undefined, undefined, {
        image: CreateImage(stack)
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/existingImage:latest"
                        ]
                    ]
                },
            }
        ],
    });
});
test('Test with custom container definition', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, { cpu: 256, memoryLimitMiB: 512 });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Cpu: '256',
        Memory: '512'
    });
});
test('Test with custom cluster props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const clusterName = 'test-value';
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, { clusterName }, defaults.fakeEcrRepoArn, undefined);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName,
    });
});
test('Test with custom Fargate Service props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const serviceName = 'test-value';
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
    });
});
test('Test with custom security group', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const groupDescription = 'Test generated security group';
    const testVpc = CreateIsolatedTestVpc(stack);
    const customSg = new ec2.SecurityGroup(stack, 'custom-sg', {
        disableInlineRules: true,
        allowAllOutbound: false,
        vpc: testVpc,
        description: groupDescription
    });
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { securityGroups: [customSg] });
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        GroupDescription: groupDescription,
    });
    expect(stack).not.toHaveResource("AWS::EC2::SecurityGroup", {
        GroupDescription: 'Construct created security group',
    });
});
test('Test no image repo or image is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    const app = () => {
        __1.CreateFargateService(stack, 'test', testVpc);
    };
    expect(app).toThrowError("Not Implemented - image without repo name and version");
});
// CheckFargateProps tests
test('Check providing existing service AND props is an error', () => {
    const props = {
        existingFargateServiceObject: { place: "holder" },
        existingImageObject: { place: "holder2" }
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide an existingFargateServiceObject, you cannot provide any props defining a new service\n");
});
test('Check providing existing image AND props is an error', () => {
    const props = {
        existingImageObject: { place: "holder" },
        ecrRepositoryArn: { place: "holder2" }
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide an existingImageObject then you cannot provide an ecrRepositoryArn nor ecrImageVersion\n");
});
test('Check providing vpc in the targetGroupsProps is an error', () => {
    const props = {
        targetGroupProps: { vpc: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("Provide all VPC info at Construct level, not within targetGroupProps\n");
});
test('Check providing taskDefinition in the fargateServiceProps is an error', () => {
    const props = {
        fargateServiceProps: { taskDefinition: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("The construct cannot accept an existing task definition in fargateServiceProps\n");
});
test('Check providing cluster in fargateServiceProps AND clusterProps is an error', () => {
    const props = {
        fargateServiceProps: { cluster: { place: "holder" } },
        clusterProps: { place: "holder2" },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide a cluster in fargateServiceProps then you cannot provide clusterProps\n");
});
test('Check providing vpc in clusterProps is an error', () => {
    const props = {
        clusterProps: { vpc: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("All services in the construct use the construct VPC, you cannot specify a VPC in clusterProps\n");
});
test('Check providing existing service without existing container and existing VPC is an error', () => {
    const props = {
        existingFargateServiceObject: { place: "holder" },
        existingVpc: { place: "holder2" },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If an existing Service is indicated by supplying either existingFargateServiceObject or existingContainerDefinitionObject, then existingFargateServiceObject, existingContainerDefinitionObject, and existingVpc must all be provided\n");
});
// Helper functions
function CreateIsolatedTestVpc(stack) {
    return defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
    });
}
function CreateImage(stack) {
    return ecs.ContainerImage.fromEcrRepository(ecr.Repository.fromRepositoryArn(stack, `test-container`, 
    // This is different than fakeEcrRepoArn because we're testing custom image
    "arn:aws:ecr:us-east-1:123456789012:repository/existingImage"), "latest");
}
//# sourceMappingURL=data:application/json;base64,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