"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("..");
// --------------------------------------------------------------
// Test deployment with role creation
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const jobRole = new aws_iam_1.Role(stack, 'CustomETLJobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    });
    const cfnJobProps = defaults.DefaultGlueJobProps(jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: 's3://fakescriptlocation/fakebucket',
        },
        role: jobRole.roleArn
    }, 'testETLJob', {});
    const database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const glueJob = defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        database,
        table: defaults.createGlueTable(stack, database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(glueJob.bucket).toBeDefined();
    expect(glueJob.bucket).toBeInstanceOf(aws_s3_1.Bucket);
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://fakescriptlocation/fakebucket"
            },
            Role: {
                "Fn::GetAtt": [
                    "CustomETLJobRole90A83A66",
                    "Arn"
                ]
            },
            GlueVersion: "2.0",
            NumberOfWorkers: 2,
            SecurityConfiguration: "testETLJob",
            WorkerType: "G.1X"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an existing Glue Job
// --------------------------------------------------------------
test('Create a Glue Job outside the construct', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'pythonshell',
            pythonVersion: '2',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    });
    const database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const glueJob = defaults.buildGlueJob(stack, {
        existingCfnJob,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database,
        table: defaults.createGlueTable(stack, database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(glueJob.bucket).not.toBeDefined();
    expect(glueJob.loggingBucket).not.toBeDefined();
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "pythonshell",
                PythonVersion: "2",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            WorkerType: "Standard",
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Provide additional parameters other than default ones
// --------------------------------------------------------------
test('Test custom deployment properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const commandName = 'glueetl';
    const cfnJobProps = {
        command: {
            name: commandName,
            pythonVersion: '3',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        numberOfWorkers: 2,
        workerType: 'Standard'
    };
    const database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database,
        table: defaults.createGlueTable(stack, database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // check if Glue Job Resource was created correctly
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "Standard",
        },
        Type: "AWS::Glue::Job"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the role is created
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazon.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            Description: "Existing role"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the security config is created
    expect(stack).toHaveResourceLike('AWS::Glue::SecurityConfiguration', {
        Properties: {
            EncryptionConfiguration: {
                JobBookmarksEncryption: {
                    JobBookmarksEncryptionMode: "CSE-KMS",
                    KmsKeyArn: {
                        "Fn::Join": [
                            "", [
                                "arn:", {
                                    Ref: "AWS::Partition",
                                },
                                ":kms:", {
                                    Ref: "AWS::Region",
                                },
                                ":", {
                                    Ref: "AWS::AccountId",
                                },
                                ":alias/aws/glue",
                            ],
                        ],
                    },
                },
                S3Encryptions: [{
                        S3EncryptionMode: "SSE-S3",
                    }],
            },
            Name: "ETLJobSecurityConfig",
        },
        Type: "AWS::Glue::SecurityConfiguration",
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Do not supply parameters and error out
// --------------------------------------------------------------
test('Do no supply glueJobProps or existingCfnJob and error out', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database,
            table: defaults.createGlueTable(stack, database, defaults.DefaultGlueTableProps(database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' }))
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Allow the construct to create the job role required
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database,
        table: defaults.createGlueTable(stack, database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com"
                        }
                    }],
                Version: "2012-10-17"
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when output location is provided
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3,
            existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                versioned: false,
                bucketName: 'outputbucket',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
            })
        },
        database,
        table: defaults.createGlueTable(stack, database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        Properties: {
            BucketEncryption: {
                ServerSideEncryptionConfiguration: [{
                        ServerSideEncryptionByDefault: {
                            SSEAlgorithm: "AES256"
                        }
                    }]
            },
            BucketName: "outputbucket"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when script location not provided - throw error
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: jobID,
            pythonVersion: '3'
        }
    };
    const database = defaults.createGlueDatabase(stack);
    try {
        defaults.buildGlueJob(stack, {
            glueJobProps: cfnJobProps,
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3,
                existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                    versioned: false
                })
            },
            database,
            table: defaults.createGlueTable(stack, database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
        expect(error.message).toEqual('Either one of CfnJob.JobCommandProperty.scriptLocation or KinesisstreamsToGluejobProps.etlCodeAsset ' +
            'has to be provided. If the ETL Job code file exists in a local filesystem, please set KinesisstreamsToGluejobProps.etlCodeAsset. ' +
            'If the ETL Job is available in an S3 bucket, set the CfnJob.JobCommandProperty.scriptLocation property');
    }
});
// --------------------------------------------------------------
// Dont pass Job Command attributes and it should throw an error
// --------------------------------------------------------------
test('Test for incorrect Job Command property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {},
            database,
            table: defaults.createGlueTable(stack, database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Check for CfnJob.JobCommandProperty type
// --------------------------------------------------------------
test('check for JobCommandProperty type', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {
                command: {
                    fakekey: 'fakevalue'
                }
            },
            database,
            table: defaults.createGlueTable(stack, database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Supply maxCapacity with GlueVersion 2.0 and error out
// --------------------------------------------------------------
test('GlueJob configuration with glueVersion 2.0 should not support maxCapacity and error out', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database,
            table: defaults.createGlueTable(stack, database, defaults.DefaultGlueTableProps(database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                glueVersion: '2.0',
                maxCapacity: '2'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Fail if setting maxCapacity and WorkerType/ NumberOfWorkers
// --------------------------------------------------------------
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database,
            table: defaults.createGlueTable(stack, database, defaults.DefaultGlueTableProps(database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                workerType: 'Standard'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database,
            table: defaults.createGlueTable(stack, database, defaults.DefaultGlueTableProps(database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        const database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database,
            table: defaults.createGlueTable(stack, database, defaults.DefaultGlueTableProps(database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2,
                workerType: 'G1.X'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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