"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const glue = require("aws-cdk-lib/aws-glue");
const iam = require("aws-cdk-lib/aws-iam");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kms = require("aws-cdk-lib/aws-kms");
const lambda = require("aws-cdk-lib/aws-lambda");
const mediastore = require("aws-cdk-lib/aws-mediastore");
const s3 = require("aws-cdk-lib/aws-s3");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const mediastore_defaults_1 = require("../lib/mediastore-defaults");
const sagemaker_helper_1 = require("../lib/sagemaker-helper");
const test_helper_1 = require("./test-helper");
test('Test with valid props', () => {
    const props = {};
    defaults.CheckProps(props);
});
test('Test fail DynamoDB table check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingTableObj: new dynamodb.Table(stack, 'placeholder', defaults.DefaultTableProps),
        dynamoTableProps: defaults.DefaultTableProps,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
test("Test fail Lambda function check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        },
        existingLambdaObj: new lambda.Function(stack, "placeholder", {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: "index.handler",
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n");
});
test("Test fail SQS Queue check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        queueProps: {},
        existingQueueObj: new sqs.Queue(stack, 'placeholder', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps or existingQueueObj, but not both.\n');
});
test('Test fail SQS queue check when queueProps.encryptionMasterKey and encryptionKey are both specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'key')
        },
        encryptionKey: new kms.Key(stack, 'otherkey')
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - Either provide queueProps.encryptionMasterKey or encryptionKey, but not both.\n');
});
test('Test fail SQS queue check when queueProps.encryptionMasterKey and encryptionKeyProps are both specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        encryptionKeyProps: {
            description: 'key description'
        },
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'key')
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide queueProps.encryptionMasterKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Dead Letter Queue check', () => {
    const props = {
        deployDeadLetterQueue: false,
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If deployDeadLetterQueue is false then deadLetterQueueProps cannot be specified.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and undefined deadLetterQueueProps', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to true and deadLetterQueueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: true },
        deadLetterQueueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false and deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with deadLetterQueueProps fifo set to true', () => {
    const props = {
        deadLetterQueueProps: { fifo: true },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test('Test fail Dead Letter Queue check with queueProps fifo set to false', () => {
    const props = {
        queueProps: { fifo: false },
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - If you specify a fifo: true in either queueProps or deadLetterQueueProps, you must also set fifo: true in the other props object. \
    Fifo must match for the Queue and the Dead Letter Queue.\n');
});
test("Test fail MediaStore container check", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const mediaStoreContainer = new mediastore.CfnContainer(stack, "placeholder", mediastore_defaults_1.MediaStoreContainerProps());
    const props = {
        mediaStoreContainerProps: mediastore_defaults_1.MediaStoreContainerProps(),
        existingMediaStoreContainerObj: mediaStoreContainer,
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide mediaStoreContainerProps or existingMediaStoreContainerObj, but not both.\n");
});
test('Test fail Kinesis stream check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const stream = new kinesis.Stream(stack, 'placeholder', {});
    const props = {
        existingStreamObj: stream,
        kinesisStreamProps: {}
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
test('Test fail S3 check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingBucketObj: test_helper_1.CreateScrapBucket(stack, {}),
        bucketProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
test('Test fail SNS topic check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check with bad topic attribute name', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n');
});
test('Test fail SNS topic check when both encryptionKey and encryptionKeyProps are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        encryptionKey: new kms.Key(stack, 'key'),
        encryptionKeyProps: {
            description: 'a description'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail SNS topic check when both topicProps.masterKey and encryptionKeyProps are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        topicProps: {
            masterKey: new kms.Key(stack, 'key')
        },
        encryptionKeyProps: {
            description: 'a description'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    expect(app).toThrowError('Error - Either provide topicProps.masterKey or encryptionKeyProps, but not both.\n');
});
test('Test fail SNS topic check when both encryptionKey and topicProps.masterKey are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        encryptionKey: new kms.Key(stack, 'key'),
        topicProps: {
            masterKey: new kms.Key(stack, 'otherkey')
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps.masterKey or encryptionKey, but not both.\n');
});
test('Test fail Glue job check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const _jobRole = new iam.Role(stack, 'CustomETLJobRole', {
        assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
    });
    const jobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: new s3.Bucket(stack, 'ScriptBucket').bucketArn,
        },
        role: new iam.Role(stack, 'JobRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com')
        }).roleArn
    }, 'testETLJob', {});
    const job = new glue.CfnJob(stack, 'placeholder', jobProps);
    const props = {
        glueJobProps: jobProps,
        existingGlueJob: job
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide glueJobProps or existingGlueJob, but not both.\n');
});
test('Test fail SageMaker endpoint check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    // Build Sagemaker Inference Endpoint
    const modelProps = {
        primaryContainer: {
            image: "<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest",
            modelDataUrl: "s3://<bucket-name>/<prefix>/model.tar.gz",
        },
    };
    const buildSagemakerEndpointResponse = sagemaker_helper_1.BuildSagemakerEndpoint(stack, { modelProps });
    const props = {
        existingSagemakerEndpointObj: buildSagemakerEndpointResponse.endpoint,
        endpointProps: {
            endpointConfigName: 'placeholder'
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide endpointProps or existingSagemakerEndpointObj, but not both.\n');
});
test('Test fail Secret check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail encryption key check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        enableKeyRotation: false
    });
    const props = {
        encryptionKey: key,
        encryptionKeyProps: {},
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide encryptionKey or encryptionKeyProps, but not both.\n');
});
test('Test fail Vpc check with deployVpc', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deployVpc: true,
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail Vpc check with vpcProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        vpcProps: defaults.DefaultPublicPrivateVpcProps(),
        existingVpc: defaults.buildVpc(stack, {
            defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
        }),
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Test fail multiple failures message', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        secretProps: {},
        existingSecretObj: defaults.buildSecretsManagerSecret(stack, 'secret', {}),
        topicProps: {},
        existingTopicObj: new sns.Topic(stack, 'placeholder', {})
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide topicProps or existingTopicObj, but not both.\n' +
        'Error - Either provide secretProps or existingSecretObj, but not both.\n');
});
test('Test fail existing log bucket and log bucket prop check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingLoggingBucketObj: new s3.Bucket(stack, 'logging-bucket'),
        loggingBucketProps: {
            autoDeleteObjects: true
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingLoggingBucketObj or loggingBucketProps, but not both.\n');
});
test('Test fail false logS3Accesslogs and loggingBucketProps check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingLoggingBucketObj: new s3.Bucket(stack, 'logging-bucket'),
        logS3AccessLogs: false
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If logS3AccessLogs is false, supplying loggingBucketProps or existingLoggingBucketObj is invalid.\n');
});
test('Test fail existingBucketObj and loggingBucketProps check', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        existingBucketObj: new s3.Bucket(stack, 'temp-bucket'),
        loggingBucketProps: {
            autoDeleteObjects: true
        }
    };
    const app = () => {
        defaults.CheckProps(props);
    };
    // Assertion
    expect(app).toThrowError('Error - If existingBucketObj is provided, supplying loggingBucketProps or logS3AccessLogs is an error.\n');
});
test('Test successful CheckListValues', () => {
    const app = () => {
        defaults.CheckListValues(['one', 'two', 'four'], ['four', 'one'], 'test value');
    };
    // Assertion
    expect(app).not.toThrowError();
});
test('Test unsuccessful CheckListValues', () => {
    const app = () => {
        defaults.CheckListValues(['one', 'two', 'four'], ['four', 'three'], 'test value');
    };
    // Assertion
    expect(app).toThrowError('Invalid test value submitted - three');
});
//# sourceMappingURL=data:application/json;base64,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