"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const kinesisanalytics = require("aws-cdk-lib/aws-kinesisanalytics");
const kinesisFirehose = require("aws-cdk-lib/aws-kinesisfirehose");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test("test kinesisanalytics override inputProperty", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const inputProperty = {
        inputSchema: {
            recordColumns: [{ name: "x", sqlType: "y" }],
            recordFormat: { recordFormatType: "csv" },
        },
        namePrefix: "zzz",
    };
    const defaultProps = defaults.DefaultCfnApplicationProps;
    const inProps = {
        inputs: [inputProperty],
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new kinesisanalytics.CfnApplication(stack, "KinesisAnalytics", outProps);
    expect(stack).toHaveResource("AWS::KinesisAnalytics::Application", {
        Inputs: [
            {
                InputSchema: {
                    RecordColumns: [
                        {
                            Name: "x",
                            SqlType: "y",
                        },
                    ],
                    RecordFormat: {
                        RecordFormatType: "csv",
                    },
                },
                NamePrefix: "zzz",
            },
        ],
    });
});
test("Test default implementation", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const newFirehose = CreateFirehose(stack);
    const kinesisProps = {
        kinesisFirehose: newFirehose,
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
    };
    defaults.buildKinesisAnalyticsApp(stack, kinesisProps);
    expect(stack).toHaveResourceLike("AWS::KinesisAnalytics::Application", {
        Inputs: [{
                InputSchema: {
                    RecordColumns: [{
                            Name: 'ts',
                            SqlType: 'TIMESTAMP',
                            Mapping: '$.timestamp'
                        }, {
                            Name: 'trip_id',
                            SqlType: 'VARCHAR(64)',
                            Mapping: '$.trip_id'
                        }],
                    RecordFormat: {
                        RecordFormatType: 'JSON'
                    },
                    RecordEncoding: 'UTF-8'
                },
                NamePrefix: 'SOURCE_SQL_STREAM'
            }]
    });
});
// test('Test for customer overrides', {
// test('Check policy created', {
function CreateFirehose(stack) {
    // Creating the Firehose is kind of a big deal. FirehoseToS3 is not readily available here in core,
    // so this routine pretty much replicates it. If this function ceases to work correctly, look at
    // FirehoseToS3 and see if that changed.
    const destinationBucket = defaults.CreateScrapBucket(stack, {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    });
    const kinesisFirehoseLogGroup = defaults.buildLogGroup(stack, "firehose-log-group", {});
    const cwLogStream = kinesisFirehoseLogGroup.addStream("firehose-log-stream");
    const firehoseRole = new iam.Role(stack, "test-role", {
        assumedBy: new iam.ServicePrincipal("firehose.amazonaws.com"),
    });
    // Setup the IAM policy for Kinesis Firehose
    const firehosePolicy = new iam.Policy(stack, "KinesisFirehosePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "s3:AbortMultipartUpload",
                    "s3:GetBucketLocation",
                    "s3:GetObject",
                    "s3:ListBucket",
                    "s3:ListBucketMultipartUploads",
                    "s3:PutObject",
                ],
                resources: [
                    `${destinationBucket.bucketArn}`,
                    `${destinationBucket.bucketArn}/*`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["logs:PutLogEvents"],
                resources: [
                    `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${kinesisFirehoseLogGroup.logGroupName}:log-stream:${cwLogStream.logStreamName}`,
                ],
            }),
        ],
    });
    // Attach policy to role
    firehosePolicy.attachToRole(firehoseRole);
    const awsManagedKey = kms.Alias.fromAliasName(stack, "aws-managed-key", "alias/aws/s3");
    const defaultKinesisFirehoseProps = defaults.DefaultCfnDeliveryStreamProps(destinationBucket.bucketArn, firehoseRole.roleArn, kinesisFirehoseLogGroup.logGroupName, cwLogStream.logStreamName, awsManagedKey);
    destinationBucket.grantPut(firehoseRole);
    const firehose = new kinesisFirehose.CfnDeliveryStream(stack, "KinesisFirehose", defaultKinesisFirehoseProps);
    return firehose;
}
//# sourceMappingURL=data:application/json;base64,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