"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    defaults.buildKinesisStream(stack, {});
    expect(stack).toHaveResourceLike('AWS::Kinesis::Stream', {
        Type: "AWS::Kinesis::Stream",
        Properties: {
            StreamEncryption: {
                EncryptionType: "KMS"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper setup
    const encKey = defaults.buildEncryptionKey(stack);
    // Helper declaration
    defaults.buildKinesisStream(stack, {
        kinesisStreamProps: {
            streamName: 'myCustomKinesisStream',
            encryption: kinesis.StreamEncryption.KMS,
            encryptionKey: encKey
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'myCustomKinesisStream'
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing stream
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper setup
    const stream = new kinesis.Stream(stack, 'existing-stream', {
        shardCount: 2,
        retentionPeriod: aws_cdk_lib_1.Duration.days(3)
    });
    // Helper declaration
    defaults.buildKinesisStream(stack, {
        existingStreamObj: stream,
        // These props will be ignored as an existing stream was provided
        kinesisStreamProps: {
            shardCount: 1,
            retentionPeriod: aws_cdk_lib_1.Duration.days(1)
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        ShardCount: 2,
        RetentionPeriodHours: 72
    });
});
test('Count Kinesis CW Alarms', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cwList = defaults.buildKinesisStreamCWAlarms(stack);
    expect(cwList.length).toEqual(2);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy1zdHJlYW1zLWhlbHBlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsia2luZXNpcy1zdHJlYW1zLWhlbHBlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxVQUFVO0FBQ1YsNkNBQThDO0FBQzlDLGdDQUFnQztBQUNoQyxtREFBbUQ7QUFDbkQsNENBQStDO0FBQy9DLGdDQUE4QjtBQUU5QixpRUFBaUU7QUFDakUsNkNBQTZDO0FBQzdDLGlFQUFpRTtBQUNqRSxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO0lBQ3RELFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUMxQixxQkFBcUI7SUFDckIsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUV2QyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCLEVBQUU7UUFDdkQsSUFBSSxFQUFFLHNCQUFzQjtRQUM1QixVQUFVLEVBQUU7WUFDVixnQkFBZ0IsRUFBRTtnQkFDaEIsY0FBYyxFQUFFLEtBQUs7YUFDdEI7U0FDRjtLQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0FBQ3RDLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLHVDQUF1QztBQUN2QyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsZUFBZTtJQUNmLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNsRCxxQkFBcUI7SUFDckIsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRTtRQUNqQyxrQkFBa0IsRUFBRTtZQUNsQixVQUFVLEVBQUUsdUJBQXVCO1lBQ25DLFVBQVUsRUFBRSxPQUFPLENBQUMsZ0JBQWdCLENBQUMsR0FBRztZQUN4QyxhQUFhLEVBQUUsTUFBTTtTQUN0QjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsc0JBQXNCLEVBQUU7UUFDbkQsSUFBSSxFQUFFLHVCQUF1QjtLQUM5QixDQUFDLENBQUM7SUFDSCxjQUFjO0lBQ2QsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxlQUFlLEVBQUU7UUFDNUMsaUJBQWlCLEVBQUUsSUFBSTtLQUN4QixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSxxQ0FBcUM7QUFDckMsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxHQUFHLEVBQUU7SUFDOUMsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLGVBQWU7SUFDZixNQUFNLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1FBQzFELFVBQVUsRUFBRSxDQUFDO1FBQ2IsZUFBZSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztLQUNsQyxDQUFDLENBQUM7SUFDSCxxQkFBcUI7SUFDckIsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRTtRQUNqQyxpQkFBaUIsRUFBRSxNQUFNO1FBRXpCLGlFQUFpRTtRQUNqRSxrQkFBa0IsRUFBRTtZQUNsQixVQUFVLEVBQUUsQ0FBQztZQUNiLGVBQWUsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDbEM7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHNCQUFzQixFQUFFO1FBQ25ELFVBQVUsRUFBRSxDQUFDO1FBQ2Isb0JBQW9CLEVBQUUsRUFBRTtLQUN6QixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLDBCQUEwQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRTFELE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ25DLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgU3RhY2ssIER1cmF0aW9uIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICcuLi8nO1xuaW1wb3J0ICogYXMga2luZXNpcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mta2luZXNpcyc7XG5pbXBvcnQgeyBSZXNvdXJjZVBhcnQgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IG1pbmltYWwgZGVwbG95bWVudCB3aXRoIG5vIHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IG1pbmltYWwgZGVwbG95bWVudCB3aXRoIG5vIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICBkZWZhdWx0cy5idWlsZEtpbmVzaXNTdHJlYW0oc3RhY2ssIHt9KTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpLaW5lc2lzOjpTdHJlYW0nLCB7XG4gICAgVHlwZTogXCJBV1M6OktpbmVzaXM6OlN0cmVhbVwiLFxuICAgIFByb3BlcnRpZXM6IHtcbiAgICAgIFN0cmVhbUVuY3J5cHRpb246IHtcbiAgICAgICAgRW5jcnlwdGlvblR5cGU6IFwiS01TXCJcbiAgICAgIH1cbiAgICB9XG4gIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgdy8gY3VzdG9tIHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gY3VzdG9tIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIC8vIEhlbHBlciBzZXR1cFxuICBjb25zdCBlbmNLZXkgPSBkZWZhdWx0cy5idWlsZEVuY3J5cHRpb25LZXkoc3RhY2spO1xuICAvLyBIZWxwZXIgZGVjbGFyYXRpb25cbiAgZGVmYXVsdHMuYnVpbGRLaW5lc2lzU3RyZWFtKHN0YWNrLCB7XG4gICAga2luZXNpc1N0cmVhbVByb3BzOiB7XG4gICAgICBzdHJlYW1OYW1lOiAnbXlDdXN0b21LaW5lc2lzU3RyZWFtJyxcbiAgICAgIGVuY3J5cHRpb246IGtpbmVzaXMuU3RyZWFtRW5jcnlwdGlvbi5LTVMsXG4gICAgICBlbmNyeXB0aW9uS2V5OiBlbmNLZXlcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6S2luZXNpczo6U3RyZWFtJywge1xuICAgIE5hbWU6ICdteUN1c3RvbUtpbmVzaXNTdHJlYW0nXG4gIH0pO1xuICAvLyBBc3NlcnRpb24gM1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OktNUzo6S2V5Jywge1xuICAgIEVuYWJsZUtleVJvdGF0aW9uOiB0cnVlXG4gIH0pO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgdy8gZXhpc3Rpbmcgc3RyZWFtXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBkZXBsb3ltZW50IHcvIGV4aXN0aW5nIHN0cmVhbScsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgLy8gSGVscGVyIHNldHVwXG4gIGNvbnN0IHN0cmVhbSA9IG5ldyBraW5lc2lzLlN0cmVhbShzdGFjaywgJ2V4aXN0aW5nLXN0cmVhbScsIHtcbiAgICBzaGFyZENvdW50OiAyLFxuICAgIHJldGVudGlvblBlcmlvZDogRHVyYXRpb24uZGF5cygzKVxuICB9KTtcbiAgLy8gSGVscGVyIGRlY2xhcmF0aW9uXG4gIGRlZmF1bHRzLmJ1aWxkS2luZXNpc1N0cmVhbShzdGFjaywge1xuICAgIGV4aXN0aW5nU3RyZWFtT2JqOiBzdHJlYW0sXG5cbiAgICAvLyBUaGVzZSBwcm9wcyB3aWxsIGJlIGlnbm9yZWQgYXMgYW4gZXhpc3Rpbmcgc3RyZWFtIHdhcyBwcm92aWRlZFxuICAgIGtpbmVzaXNTdHJlYW1Qcm9wczoge1xuICAgICAgc2hhcmRDb3VudDogMSxcbiAgICAgIHJldGVudGlvblBlcmlvZDogRHVyYXRpb24uZGF5cygxKVxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpLaW5lc2lzOjpTdHJlYW0nLCB7XG4gICAgU2hhcmRDb3VudDogMixcbiAgICBSZXRlbnRpb25QZXJpb2RIb3VyczogNzJcbiAgfSk7XG59KTtcblxudGVzdCgnQ291bnQgS2luZXNpcyBDVyBBbGFybXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgY3dMaXN0ID0gZGVmYXVsdHMuYnVpbGRLaW5lc2lzU3RyZWFtQ1dBbGFybXMoc3RhY2spO1xuXG4gIGV4cGVjdChjd0xpc3QubGVuZ3RoKS50b0VxdWFsKDIpO1xufSk7Il19