"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("../index");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
require("@aws-cdk/assert/jest");
const aws_cdk_lib_1 = require("aws-cdk-lib");
test('test DynamoEventSourceProps with defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(aws_cdk_lib_1.Duration.hours(24));
});
test('test DynamoEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.DynamoEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test DynamoEventSourceProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.DynamoEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test S3EventSourceProps w/ default props', () => {
    const props = defaults.S3EventSourceProps();
    expect(props).toEqual({
        events: ["s3:ObjectCreated:*"]
    });
});
test('test S3EventSourceProps w/ user props', () => {
    const s3EventSourceProps = {
        events: [
            s3.EventType.OBJECT_CREATED_POST
        ]
    };
    const props = defaults.S3EventSourceProps(s3EventSourceProps);
    expect(props).toEqual({
        events: ["s3:ObjectCreated:Post"]
    });
});
test('test KinesisEventSourceProps with defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack);
    expect(props.onFailure).toBeInstanceOf(aws_lambda_event_sources_1.SqsDlq);
    expect(props.startingPosition).toEqual("TRIM_HORIZON");
    expect(props.bisectBatchOnError).toEqual(true);
    expect(props.retryAttempts).toEqual(500);
    expect(props.maxRecordAge).toEqual(aws_cdk_lib_1.Duration.hours(24));
});
test('test KinesisEventSourceProps with deployDeadLetterQueue: false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = defaults.KinesisEventSourceProps(stack, {
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        startingPosition: "TRIM_HORIZON",
        bisectBatchOnError: true,
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        retryAttempts: 500
    });
});
test('test KinesisEventSourceProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myProps = {
        startingPosition: lambda.StartingPosition.LATEST,
        batchSize: 5,
        retryAttempts: 3
    };
    const props = defaults.KinesisEventSourceProps(stack, {
        eventSourceProps: myProps,
        deploySqsDlqQueue: false
    });
    expect(props).toEqual({
        batchSize: 5,
        startingPosition: "LATEST",
        maxRecordAge: aws_cdk_lib_1.Duration.hours(24),
        bisectBatchOnError: true,
        retryAttempts: 3
    });
});
test('test sqsDlqQueueProps override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.KinesisEventSourceProps(stack, {
        sqsDlqQueueProps: {
            queueName: 'hello-world',
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(50)
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "hello-world",
        VisibilityTimeout: 50
    });
});
//# sourceMappingURL=data:application/json;base64,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