"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const ec2 = require("aws-cdk-lib/aws-ec2");
function buildTestOpenSearchDomain(stack, openSearchDomainName, clientDomainProps, lambdaRoleARN, vpc) {
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    const cognitoAuthorizedRole = defaults.setupCognitoForSearchService(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    return defaults.buildOpenSearch(stack, {
        userpool,
        identitypool,
        cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
        serviceRoleARN: lambdaRoleARN ? lambdaRoleARN : undefined,
        vpc,
        openSearchDomainName,
        clientDomainProps
    });
}
function deployStack() {
    return new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
}
test('Test override SnapshotOptions for buildOpenSearch', () => {
    const stack = deployStack();
    const buildOpenSearchResponse = buildTestOpenSearchDomain(stack, 'test-domain', {
        snapshotOptions: {
            automatedSnapshotStartHour: 5
        }
    });
    expect(buildOpenSearchResponse.domain).toBeDefined();
    expect(buildOpenSearchResponse.role).toBeDefined();
    expect(stack).toHaveResource('AWS::OpenSearchService::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoDashboardConfigureRoleEC5F4809",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        EngineVersion: "OpenSearch_1.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 5
        }
    });
});
test('Test VPC with 1 AZ, Zone Awareness Disabled', () => {
    const stack = deployStack();
    const vpc = defaults.getTestVpc(stack, false);
    buildTestOpenSearchDomain(stack, 'test-domain', {
        clusterConfig: {
            dedicatedMasterEnabled: true,
            dedicatedMasterCount: 3,
            instanceCount: 3,
            zoneAwarenessEnabled: false
        }
    }, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false
        }
    });
});
test('Test VPC with 2 AZ, Zone Awareness Enabled', () => {
    // If no environment is specified, a VPC will use 2 AZs by default.
    // If an environment is specified, a VPC will use 3 AZs by default.
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {});
    const vpc = defaults.getTestVpc(stack, false);
    const buildOpenSearchResponse = buildTestOpenSearchDomain(stack, 'test-domain', {}, undefined, vpc);
    expect(buildOpenSearchResponse.domain).toBeDefined();
    expect(buildOpenSearchResponse.role).toBeDefined();
    expect(stack).toHaveResourceLike('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test VPC with 3 AZ, Zone Awareness Enabled', () => {
    // If no environment is specified, a VPC will use 2 AZs by default.
    // If an environment is specified, a VPC will use 3 AZs by default.
    const stack = deployStack();
    const vpc = defaults.getTestVpc(stack);
    buildTestOpenSearchDomain(stack, 'test-domain', {}, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test deployment with an existing private VPC', () => {
    const stack = deployStack();
    const vpc = new ec2.Vpc(stack, 'existing-private-vpc-test', {
        natGateways: 1,
        subnetConfiguration: [
            {
                cidrMask: 24,
                name: 'application',
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            },
            {
                cidrMask: 24,
                name: "public",
                subnetType: ec2.SubnetType.PUBLIC,
            }
        ]
    });
    buildTestOpenSearchDomain(stack, 'test-domain', {}, undefined, vpc);
    expect(stack).toHaveResourceLike('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: true
        }
    });
});
test('Test error thrown with no private subnet configurations', () => {
    const stack = deployStack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: {
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: "public",
                    subnetType: ec2.SubnetType.PUBLIC,
                }
            ]
        }
    });
    const app = () => {
        buildTestOpenSearchDomain(stack, 'test-domain', {}, undefined, vpc);
    };
    expect(app).toThrowError('Error - No isolated or private subnets available in VPC');
});
test('Test engine version override for buildOpenSearch', () => {
    const stack = deployStack();
    buildTestOpenSearchDomain(stack, 'test-domain', {
        engineVersion: 'OpenSearch_1.0'
    });
    expect(stack).toHaveResource('AWS::OpenSearchService::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoDashboardConfigureRoleEC5F4809",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        EngineVersion: "OpenSearch_1.0",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Test deployment with lambdaRoleARN', () => {
    const stack = deployStack();
    const buildOpenSearchResponse = buildTestOpenSearchDomain(stack, 'test-domain', {}, 'arn:aws:us-east-1:mylambdaRoleARN');
    expect(buildOpenSearchResponse.domain).toBeDefined();
    expect(buildOpenSearchResponse.role).toBeDefined();
    expect(stack).toHaveResource('AWS::OpenSearchService::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: [
                            {
                                "Fn::GetAtt": [
                                    "CognitoAuthorizedRole14E74FE0",
                                    "Arn"
                                ]
                            },
                            "arn:aws:us-east-1:mylambdaRoleARN"
                        ]
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoDashboardConfigureRoleEC5F4809",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        EngineVersion: "OpenSearch_1.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Count OpenSearch CloudWatch alarms', () => {
    const stack = new aws_cdk_lib_1.Stack();
    buildTestOpenSearchDomain(stack, 'test-domain');
    const cwList = defaults.buildOpenSearchCWAlarms(stack);
    expect(cwList.length).toEqual(9);
});
//# sourceMappingURL=data:application/json;base64,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