"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const s3n = require("aws-cdk-lib/aws-s3-notifications");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
test('check exception for Missing existingBucketObj from props for deploy = false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    try {
        defaults.buildS3Bucket(stack, {});
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('s3 bucket with bucketId', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {}, 'my');
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "myS3LoggingBucketDE461344"
            }
        },
    }));
});
test('s3 bucket with bucketProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'mybucket'
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketName: "mybucket"
    }));
});
test('s3 bucket with default props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {});
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
        VersioningConfiguration: {
            Status: "Enabled"
        }
    }));
});
test('s3 bucket with life cycle policy', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            lifecycleRules: [{
                    expiration: aws_cdk_lib_1.Duration.days(365),
                    transitions: [{
                            storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30)
                        }, {
                            storageClass: aws_s3_1.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90)
                        }]
                }]
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    ExpirationInDays: 365,
                    Status: "Enabled",
                    Transitions: [
                        {
                            StorageClass: "STANDARD_IA",
                            TransitionInDays: 30
                        },
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ]
                }
            ]
        }
    }));
});
test('s3 bucket with access logging configured', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    // This value should be populated, entered Issue 907
    // expect(response.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            }
        },
    }));
});
test('Check S3 Bucket policy', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false",
                        },
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "S3Bucket07682993",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "S3Bucket07682993",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17",
        },
    }));
});
test('s3 bucket with LoggingBucket and versioning turned off', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket,
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    // The line below fails, this appears to be a bug. Entered Issue 906
    //  expect(response.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket versioning turned off', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket with LoggingBucket and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        loggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    }));
    assert_1.expect(stack).to(assert_1.haveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "S3LoggingBucket800A2B27"
        }
    }));
});
test('s3 bucket versioning turned on', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
        }
    });
    expect(buildS3BucketResponse.bucket).toBeDefined();
    expect(buildS3BucketResponse.loggingBucket).toBeDefined();
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LifecycleConfiguration: {
            Rules: [
                {
                    NoncurrentVersionTransitions: [
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ],
                    Status: "Enabled"
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
        VersioningConfiguration: {
            Status: "Enabled"
        }
    }));
});
test('Suppress cfn-nag warning for s3 bucket notification', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {});
    const buildQueueResponse = defaults.buildQueue(stack, "S3BucketNotificationQueue", {});
    buildS3BucketResponse.bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.SqsDestination(buildQueueResponse.queue));
    defaults.addCfnNagS3BucketNotificationRulesToSuppress(stack, "BucketNotificationsHandler050a0587b7544547bf325f094a3db834");
    assert_1.expect(stack).to(assert_1.haveResource("AWS::Lambda::Function", {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W58",
                        reason: "Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions."
                    },
                    {
                        id: 'W89',
                        reason: `This is not a rule for the general case, just for specific use cases/industries`
                    },
                    {
                        id: 'W92',
                        reason: `Impossible for us to define the correct concurrency for clients`
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResource("AWS::IAM::Policy", {
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "Bucket resource is '*' due to circular dependency with bucket and role creation at the same time"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('test s3Bucket removalPolicy override', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
    }, 'test-bucket');
    expect(stack).toHaveResourceLike("AWS::S3::Bucket", {
        Type: 'AWS::S3::Bucket',
        Properties: {
            AccessControl: "LogDeliveryWrite"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('s3 bucket with logging turned off', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const respbuildS3BucketResponsense = defaults.buildS3Bucket(stack, {
        logS3AccessLogs: false,
    });
    expect(respbuildS3BucketResponsense.bucket).toBeDefined();
    expect(respbuildS3BucketResponsense.loggingBucket).not.toBeDefined();
    expect(stack).not.toHaveResourceLike("AWS::S3::Bucket", {
        LoggingConfiguration: {},
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtYnVja2V0LWhlbHBlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiczMtYnVja2V0LWhlbHBlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCw0Q0FBa0Y7QUFDbEYsNkNBQTZEO0FBQzdELHlDQUF5QztBQUN6Qyx3REFBd0Q7QUFDeEQscUNBQXFDO0FBQ3JDLGdDQUE4QjtBQUM5QiwrQ0FBMEQ7QUFFMUQsSUFBSSxDQUFDLDZFQUE2RSxFQUFFLEdBQUcsRUFBRTtJQUN2RixNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixJQUFJO1FBQ0YsUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDbkM7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDakM7QUFDSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxxQkFBcUIsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFFdEUsTUFBTSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUUxRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7UUFDbEQsb0JBQW9CLEVBQUU7WUFDcEIscUJBQXFCLEVBQUU7Z0JBQ3JCLEdBQUcsRUFBRSwyQkFBMkI7YUFDakM7U0FDRjtLQUNGLENBQUMsQ0FBQyxDQUFDO0FBQ04sQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO0lBQ3RDLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0scUJBQXFCLEdBQUcsUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUU7UUFDMUQsV0FBVyxFQUFFO1lBQ1gsVUFBVSxFQUFFLFVBQVU7U0FDdkI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDbkQsTUFBTSxDQUFDLHFCQUFxQixDQUFDLGFBQWEsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBRTFELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxVQUFVLEVBQUUsVUFBVTtLQUN2QixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtJQUN4QyxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBRWhFLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNuRCxNQUFNLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFFMUQsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGlCQUFpQixFQUFFO1FBQ2xELGdCQUFnQixFQUFFO1lBQ2hCLGlDQUFpQyxFQUFFO2dCQUNqQztvQkFDRSw2QkFBNkIsRUFBRTt3QkFDN0IsWUFBWSxFQUFFLFFBQVE7cUJBQ3ZCO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELDhCQUE4QixFQUFFO1lBQzlCLGVBQWUsRUFBRSxJQUFJO1lBQ3JCLGlCQUFpQixFQUFFLElBQUk7WUFDdkIsZ0JBQWdCLEVBQUUsSUFBSTtZQUN0QixxQkFBcUIsRUFBRSxJQUFJO1NBQzVCO1FBQ0QsdUJBQXVCLEVBQUU7WUFDdkIsTUFBTSxFQUFFLFNBQVM7U0FDbEI7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtDQUFrQyxFQUFFLEdBQUcsRUFBRTtJQUM1QyxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFO1FBQzFELFdBQVcsRUFBRTtZQUNYLGNBQWMsRUFBRSxDQUFDO29CQUNmLFVBQVUsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7b0JBQzlCLFdBQVcsRUFBRSxDQUFDOzRCQUNaLFlBQVksRUFBRSxxQkFBWSxDQUFDLGlCQUFpQjs0QkFDNUMsZUFBZSxFQUFFLHNCQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQzt5QkFDbkMsRUFBRTs0QkFDRCxZQUFZLEVBQUUscUJBQVksQ0FBQyxPQUFPOzRCQUNsQyxlQUFlLEVBQUUsc0JBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO3lCQUNuQyxDQUFDO2lCQUNILENBQUM7U0FDSDtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNuRCxNQUFNLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFFMUQsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGlCQUFpQixFQUFFO1FBQ2xELHNCQUFzQixFQUFFO1lBQ3RCLEtBQUssRUFBRTtnQkFDTDtvQkFDRSxnQkFBZ0IsRUFBRSxHQUFHO29CQUNyQixNQUFNLEVBQUUsU0FBUztvQkFDakIsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLFlBQVksRUFBRSxhQUFhOzRCQUMzQixnQkFBZ0IsRUFBRSxFQUFFO3lCQUNyQjt3QkFDRDs0QkFDRSxZQUFZLEVBQUUsU0FBUzs0QkFDdkIsZ0JBQWdCLEVBQUUsRUFBRTt5QkFDckI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxHQUFHLEVBQUU7SUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtRQUM3QyxzQkFBc0IsRUFBRSxJQUFJLGVBQU0sQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUUsRUFBRSxDQUFDO0tBQ25FLENBQUMsQ0FBQztJQUVILE1BQU0scUJBQXFCLEdBQUcsUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUU7UUFDMUQsV0FBVyxFQUFFO1lBQ1gsc0JBQXNCLEVBQUUsUUFBUTtTQUNqQztLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNuRCxvREFBb0Q7SUFDcEQsZ0RBQWdEO0lBRWhELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxvQkFBb0IsRUFBRTtZQUNwQixxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGtCQUFrQjthQUN4QjtTQUNGO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7SUFDbEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFbEMsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FDakIscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtRQUNwQyxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFLE1BQU07b0JBQ2QsU0FBUyxFQUFFO3dCQUNULElBQUksRUFBRTs0QkFDSixxQkFBcUIsRUFBRSxPQUFPO3lCQUMvQjtxQkFDRjtvQkFDRCxNQUFNLEVBQUUsTUFBTTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1QsR0FBRyxFQUFFLEdBQUc7cUJBQ1Q7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSOzRCQUNFLFlBQVksRUFBRTtnQ0FDWixrQkFBa0I7Z0NBQ2xCLEtBQUs7NkJBQ047eUJBQ0Y7d0JBQ0Q7NEJBQ0UsVUFBVSxFQUFFO2dDQUNWLEVBQUU7Z0NBQ0Y7b0NBQ0U7d0NBQ0UsWUFBWSxFQUFFOzRDQUNaLGtCQUFrQjs0Q0FDbEIsS0FBSzt5Q0FDTjtxQ0FDRjtvQ0FDRCxJQUFJO2lDQUNMOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsWUFBWTtTQUN0QjtLQUNGLENBQUMsQ0FDSCxDQUFDO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsR0FBRyxFQUFFO0lBQ2xFLE1BQU0sS0FBSyxHQUFHLElBQUksbUJBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sUUFBUSxHQUFHLElBQUksZUFBTSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDN0Msc0JBQXNCLEVBQUUsSUFBSSxlQUFNLENBQUMsS0FBSyxFQUFFLG1CQUFtQixFQUFFLEVBQUUsQ0FBQztLQUNuRSxDQUFDLENBQUM7SUFFSCxNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFO1FBQzFELFdBQVcsRUFBRTtZQUNYLHNCQUFzQixFQUFFLFFBQVE7WUFDaEMsc0JBQXNCLEVBQUUsYUFBYTtZQUNyQyxTQUFTLEVBQUUsS0FBSztTQUNqQjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNuRCxvRUFBb0U7SUFDcEUsaURBQWlEO0lBRWpELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxnQkFBZ0IsRUFBRTtZQUNoQixpQ0FBaUMsRUFBRTtnQkFDakM7b0JBQ0UsNkJBQTZCLEVBQUU7d0JBQzdCLFlBQVksRUFBRSxRQUFRO3FCQUN2QjtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxvQkFBb0IsRUFBRTtZQUNwQixxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLGtCQUFrQjthQUN4QjtZQUNELGFBQWEsRUFBRSxhQUFhO1NBQzdCO1FBQ0QsOEJBQThCLEVBQUU7WUFDOUIsZUFBZSxFQUFFLElBQUk7WUFDckIsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLHFCQUFxQixFQUFFLElBQUk7U0FDNUI7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMzQyxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFO1FBQzFELFdBQVcsRUFBRTtZQUNYLHNCQUFzQixFQUFFLGFBQWE7WUFDckMsU0FBUyxFQUFFLEtBQUs7U0FDakI7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDbkQsTUFBTSxDQUFDLHFCQUFxQixDQUFDLGFBQWEsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBRTFELGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxpQkFBaUIsRUFBRTtRQUNsRCxnQkFBZ0IsRUFBRTtZQUNoQixpQ0FBaUMsRUFBRTtnQkFDakM7b0JBQ0UsNkJBQTZCLEVBQUU7d0JBQzdCLFlBQVksRUFBRSxRQUFRO3FCQUN2QjtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxvQkFBb0IsRUFBRTtZQUNwQixxQkFBcUIsRUFBRTtnQkFDckIsR0FBRyxFQUFFLHlCQUF5QjthQUMvQjtZQUNELGFBQWEsRUFBRSxhQUFhO1NBQzdCO1FBQ0QsOEJBQThCLEVBQUU7WUFDOUIsZUFBZSxFQUFFLElBQUk7WUFDckIsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLHFCQUFxQixFQUFFLElBQUk7U0FDNUI7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtJQUNoRSxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFO1FBQzFELGtCQUFrQixFQUFFO1lBQ2xCLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87WUFDcEMsaUJBQWlCLEVBQUUsSUFBSTtTQUN4QjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUNuRCxNQUFNLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFFMUQsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGlCQUFpQixFQUFFO1FBQ2xELGFBQWEsRUFBRSxrQkFBa0I7S0FDbEMsQ0FBQyxDQUFDLENBQUM7SUFFSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7UUFDOUQsWUFBWSxFQUFFO1lBQ1osWUFBWSxFQUFFO2dCQUNaLGdFQUFnRTtnQkFDaEUsS0FBSzthQUNOO1NBQ0Y7UUFDRCxVQUFVLEVBQUU7WUFDVixHQUFHLEVBQUUseUJBQXlCO1NBQy9CO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7SUFDMUMsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxxQkFBcUIsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRTtRQUMxRCxXQUFXLEVBQUU7WUFDWCxzQkFBc0IsRUFBRSxhQUFhO1NBQ3RDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ25ELE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUUxRCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLEVBQUU7UUFDbEQsZ0JBQWdCLEVBQUU7WUFDaEIsaUNBQWlDLEVBQUU7Z0JBQ2pDO29CQUNFLDZCQUE2QixFQUFFO3dCQUM3QixZQUFZLEVBQUUsUUFBUTtxQkFDdkI7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0Qsc0JBQXNCLEVBQUU7WUFDdEIsS0FBSyxFQUFFO2dCQUNMO29CQUNFLDRCQUE0QixFQUFFO3dCQUM1Qjs0QkFDRSxZQUFZLEVBQUUsU0FBUzs0QkFDdkIsZ0JBQWdCLEVBQUUsRUFBRTt5QkFDckI7cUJBQ0Y7b0JBQ0QsTUFBTSxFQUFFLFNBQVM7aUJBQ2xCO2FBQ0Y7U0FDRjtRQUNELG9CQUFvQixFQUFFO1lBQ3BCLHFCQUFxQixFQUFFO2dCQUNyQixHQUFHLEVBQUUseUJBQXlCO2FBQy9CO1lBQ0QsYUFBYSxFQUFFLGFBQWE7U0FDN0I7UUFDRCw4QkFBOEIsRUFBRTtZQUM5QixlQUFlLEVBQUUsSUFBSTtZQUNyQixpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLGdCQUFnQixFQUFFLElBQUk7WUFDdEIscUJBQXFCLEVBQUUsSUFBSTtTQUM1QjtRQUNELHVCQUF1QixFQUFFO1lBQ3ZCLE1BQU0sRUFBRSxTQUFTO1NBQ2xCO0tBQ0YsQ0FBQyxDQUFDLENBQUM7QUFDTixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxxREFBcUQsRUFBRSxHQUFHLEVBQUU7SUFDL0QsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxxQkFBcUIsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNoRSxNQUFNLGtCQUFrQixHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLDJCQUEyQixFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZGLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUNqSSxRQUFRLENBQUMsNENBQTRDLENBQUMsS0FBSyxFQUFFLDREQUE0RCxDQUFDLENBQUM7SUFFM0gsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO1FBQ3hELFFBQVEsRUFBRTtZQUNSLE9BQU8sRUFBRTtnQkFDUCxpQkFBaUIsRUFBRTtvQkFDakI7d0JBQ0UsRUFBRSxFQUFFLEtBQUs7d0JBQ1QsTUFBTSxFQUFFLG9NQUFvTTtxQkFDN007b0JBQ0Q7d0JBQ0UsRUFBRSxFQUFFLEtBQUs7d0JBQ1QsTUFBTSxFQUFFLGlGQUFpRjtxQkFDMUY7b0JBQ0Q7d0JBQ0UsRUFBRSxFQUFFLEtBQUs7d0JBQ1QsTUFBTSxFQUFFLGlFQUFpRTtxQkFDMUU7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztJQUVyQyxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7UUFDbkQsUUFBUSxFQUFFO1lBQ1IsT0FBTyxFQUFFO2dCQUNQLGlCQUFpQixFQUFFO29CQUNqQjt3QkFDRSxFQUFFLEVBQUUsS0FBSzt3QkFDVCxNQUFNLEVBQUUsa0dBQWtHO3FCQUMzRztpQkFDRjthQUNGO1NBQ0Y7S0FDRixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO0FBQ3ZDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxNQUFNLEtBQUssR0FBRyxJQUFJLG1CQUFLLEVBQUUsQ0FBQztJQUUxQixRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRTtRQUM1QixXQUFXLEVBQUU7WUFDWCxhQUFhLEVBQUUsMkJBQWEsQ0FBQyxPQUFPO1NBQ3JDO0tBQ0YsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUVsQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsaUJBQWlCLEVBQUU7UUFDbEQsSUFBSSxFQUFFLGlCQUFpQjtRQUN2QixVQUFVLEVBQUU7WUFDVixhQUFhLEVBQUUsa0JBQWtCO1NBQ2xDO1FBQ0QsbUJBQW1CLEVBQUUsUUFBUTtRQUM3QixjQUFjLEVBQUUsUUFBUTtLQUN6QixFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQztBQUN0QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7SUFDN0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxtQkFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSw0QkFBNEIsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRTtRQUNqRSxlQUFlLEVBQUUsS0FBSztLQUN2QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsNEJBQTRCLENBQUMsTUFBTSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDMUQsTUFBTSxDQUFDLDRCQUE0QixDQUFDLGFBQWEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUVyRSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLGlCQUFpQixFQUFFO1FBQ3RELG9CQUFvQixFQUFFLEVBQ3JCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBleHBlY3QgYXMgZXhwZWN0Q0RLLCBoYXZlUmVzb3VyY2UsIFJlc291cmNlUGFydCB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgU3RhY2ssIFJlbW92YWxQb2xpY3kgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0ICogYXMgczNuIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMy1ub3RpZmljYXRpb25zJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJy4uL2luZGV4JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgQnVja2V0LCBTdG9yYWdlQ2xhc3MgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuXG50ZXN0KCdjaGVjayBleGNlcHRpb24gZm9yIE1pc3NpbmcgZXhpc3RpbmdCdWNrZXRPYmogZnJvbSBwcm9wcyBmb3IgZGVwbG95ID0gZmFsc2UnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgdHJ5IHtcbiAgICBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7fSk7XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICBleHBlY3QoZSkudG9CZUluc3RhbmNlT2YoRXJyb3IpO1xuICB9XG59KTtcblxudGVzdCgnczMgYnVja2V0IHdpdGggYnVja2V0SWQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgYnVpbGRTM0J1Y2tldFJlc3BvbnNlID0gZGVmYXVsdHMuYnVpbGRTM0J1Y2tldChzdGFjaywge30sICdteScpO1xuXG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UuYnVja2V0KS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QoYnVpbGRTM0J1Y2tldFJlc3BvbnNlLmxvZ2dpbmdCdWNrZXQpLnRvQmVEZWZpbmVkKCk7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OlMzOjpCdWNrZXRcIiwge1xuICAgIExvZ2dpbmdDb25maWd1cmF0aW9uOiB7XG4gICAgICBEZXN0aW5hdGlvbkJ1Y2tldE5hbWU6IHtcbiAgICAgICAgUmVmOiBcIm15UzNMb2dnaW5nQnVja2V0REU0NjEzNDRcIlxuICAgICAgfVxuICAgIH0sXG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdzMyBidWNrZXQgd2l0aCBidWNrZXRQcm9wcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWlsZFMzQnVja2V0UmVzcG9uc2UgPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7XG4gICAgYnVja2V0UHJvcHM6IHtcbiAgICAgIGJ1Y2tldE5hbWU6ICdteWJ1Y2tldCdcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UuYnVja2V0KS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QoYnVpbGRTM0J1Y2tldFJlc3BvbnNlLmxvZ2dpbmdCdWNrZXQpLnRvQmVEZWZpbmVkKCk7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OlMzOjpCdWNrZXRcIiwge1xuICAgIEJ1Y2tldE5hbWU6IFwibXlidWNrZXRcIlxuICB9KSk7XG59KTtcblxudGVzdCgnczMgYnVja2V0IHdpdGggZGVmYXVsdCBwcm9wcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWlsZFMzQnVja2V0UmVzcG9uc2UgPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7fSk7XG5cbiAgZXhwZWN0KGJ1aWxkUzNCdWNrZXRSZXNwb25zZS5idWNrZXQpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UubG9nZ2luZ0J1Y2tldCkudG9CZURlZmluZWQoKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6UzM6OkJ1Y2tldFwiLCB7XG4gICAgQnVja2V0RW5jcnlwdGlvbjoge1xuICAgICAgU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBTZXJ2ZXJTaWRlRW5jcnlwdGlvbkJ5RGVmYXVsdDoge1xuICAgICAgICAgICAgU1NFQWxnb3JpdGhtOiBcIkFFUzI1NlwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSxcbiAgICBQdWJsaWNBY2Nlc3NCbG9ja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgIEJsb2NrUHVibGljQWNsczogdHJ1ZSxcbiAgICAgIEJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgSWdub3JlUHVibGljQWNsczogdHJ1ZSxcbiAgICAgIFJlc3RyaWN0UHVibGljQnVja2V0czogdHJ1ZVxuICAgIH0sXG4gICAgVmVyc2lvbmluZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgIFN0YXR1czogXCJFbmFibGVkXCJcbiAgICB9XG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdzMyBidWNrZXQgd2l0aCBsaWZlIGN5Y2xlIHBvbGljeScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWlsZFMzQnVja2V0UmVzcG9uc2UgPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7XG4gICAgYnVja2V0UHJvcHM6IHtcbiAgICAgIGxpZmVjeWNsZVJ1bGVzOiBbe1xuICAgICAgICBleHBpcmF0aW9uOiBEdXJhdGlvbi5kYXlzKDM2NSksXG4gICAgICAgIHRyYW5zaXRpb25zOiBbe1xuICAgICAgICAgIHN0b3JhZ2VDbGFzczogU3RvcmFnZUNsYXNzLklORlJFUVVFTlRfQUNDRVNTLFxuICAgICAgICAgIHRyYW5zaXRpb25BZnRlcjogRHVyYXRpb24uZGF5cygzMClcbiAgICAgICAgfSwge1xuICAgICAgICAgIHN0b3JhZ2VDbGFzczogU3RvcmFnZUNsYXNzLkdMQUNJRVIsXG4gICAgICAgICAgdHJhbnNpdGlvbkFmdGVyOiBEdXJhdGlvbi5kYXlzKDkwKVxuICAgICAgICB9XVxuICAgICAgfV1cbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UuYnVja2V0KS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QoYnVpbGRTM0J1Y2tldFJlc3BvbnNlLmxvZ2dpbmdCdWNrZXQpLnRvQmVEZWZpbmVkKCk7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OlMzOjpCdWNrZXRcIiwge1xuICAgIExpZmVjeWNsZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgIFJ1bGVzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBFeHBpcmF0aW9uSW5EYXlzOiAzNjUsXG4gICAgICAgICAgU3RhdHVzOiBcIkVuYWJsZWRcIixcbiAgICAgICAgICBUcmFuc2l0aW9uczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBTdG9yYWdlQ2xhc3M6IFwiU1RBTkRBUkRfSUFcIixcbiAgICAgICAgICAgICAgVHJhbnNpdGlvbkluRGF5czogMzBcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFN0b3JhZ2VDbGFzczogXCJHTEFDSUVSXCIsXG4gICAgICAgICAgICAgIFRyYW5zaXRpb25JbkRheXM6IDkwXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfVxuICB9KSk7XG59KTtcblxudGVzdCgnczMgYnVja2V0IHdpdGggYWNjZXNzIGxvZ2dpbmcgY29uZmlndXJlZCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgbXlidWNrZXQgPSBuZXcgQnVja2V0KHN0YWNrLCAnbXlidWNrZXQnLCB7XG4gICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogbmV3IEJ1Y2tldChzdGFjaywgJ215YWNjZXNzbG9nYnVja2V0Jywge30pXG4gIH0pO1xuXG4gIGNvbnN0IGJ1aWxkUzNCdWNrZXRSZXNwb25zZSA9IGRlZmF1bHRzLmJ1aWxkUzNCdWNrZXQoc3RhY2ssIHtcbiAgICBidWNrZXRQcm9wczoge1xuICAgICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogbXlidWNrZXRcbiAgICB9XG4gIH0pO1xuXG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UuYnVja2V0KS50b0JlRGVmaW5lZCgpO1xuICAvLyBUaGlzIHZhbHVlIHNob3VsZCBiZSBwb3B1bGF0ZWQsIGVudGVyZWQgSXNzdWUgOTA3XG4gIC8vIGV4cGVjdChyZXNwb25zZS5sb2dnaW5nQnVja2V0KS50b0JlRGVmaW5lZCgpO1xuXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpTMzo6QnVja2V0XCIsIHtcbiAgICBMb2dnaW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgRGVzdGluYXRpb25CdWNrZXROYW1lOiB7XG4gICAgICAgIFJlZjogXCJteWJ1Y2tldDE2MEY4MTMyXCJcbiAgICAgIH1cbiAgICB9LFxuICB9KSk7XG59KTtcblxudGVzdCgnQ2hlY2sgUzMgQnVja2V0IHBvbGljeScsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgZGVmYXVsdHMuYnVpbGRTM0J1Y2tldChzdGFjaywge30pO1xuXG4gIGV4cGVjdENESyhzdGFjaykudG8oXG4gICAgaGF2ZVJlc291cmNlKFwiQVdTOjpTMzo6QnVja2V0UG9saWN5XCIsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogXCJzMzoqXCIsXG4gICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgQm9vbDoge1xuICAgICAgICAgICAgICAgIFwiYXdzOlNlY3VyZVRyYW5zcG9ydFwiOiBcImZhbHNlXCIsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRWZmZWN0OiBcIkRlbnlcIixcbiAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICBBV1M6IFwiKlwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUmVzb3VyY2U6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgICAgICBcIlMzQnVja2V0MDc2ODI5OTNcIixcbiAgICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgICAgICAgICAgXCJTM0J1Y2tldDA3NjgyOTkzXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBcIi8qXCJcbiAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiLFxuICAgICAgfSxcbiAgICB9KVxuICApO1xufSk7XG5cbnRlc3QoJ3MzIGJ1Y2tldCB3aXRoIExvZ2dpbmdCdWNrZXQgYW5kIHZlcnNpb25pbmcgdHVybmVkIG9mZicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgY29uc3QgbXlidWNrZXQgPSBuZXcgQnVja2V0KHN0YWNrLCAnbXlidWNrZXQnLCB7XG4gICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogbmV3IEJ1Y2tldChzdGFjaywgJ215YWNjZXNzbG9nYnVja2V0Jywge30pXG4gIH0pO1xuXG4gIGNvbnN0IGJ1aWxkUzNCdWNrZXRSZXNwb25zZSA9IGRlZmF1bHRzLmJ1aWxkUzNCdWNrZXQoc3RhY2ssIHtcbiAgICBidWNrZXRQcm9wczoge1xuICAgICAgc2VydmVyQWNjZXNzTG9nc0J1Y2tldDogbXlidWNrZXQsXG4gICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OiAnYWNjZXNzLWxvZ3MnLFxuICAgICAgdmVyc2lvbmVkOiBmYWxzZVxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KGJ1aWxkUzNCdWNrZXRSZXNwb25zZS5idWNrZXQpLnRvQmVEZWZpbmVkKCk7XG4gIC8vIFRoZSBsaW5lIGJlbG93IGZhaWxzLCB0aGlzIGFwcGVhcnMgdG8gYmUgYSBidWcuIEVudGVyZWQgSXNzdWUgOTA2XG4gIC8vICBleHBlY3QocmVzcG9uc2UubG9nZ2luZ0J1Y2tldCkudG9CZURlZmluZWQoKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6UzM6OkJ1Y2tldFwiLCB7XG4gICAgQnVja2V0RW5jcnlwdGlvbjoge1xuICAgICAgU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBTZXJ2ZXJTaWRlRW5jcnlwdGlvbkJ5RGVmYXVsdDoge1xuICAgICAgICAgICAgU1NFQWxnb3JpdGhtOiBcIkFFUzI1NlwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSxcbiAgICBMb2dnaW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgRGVzdGluYXRpb25CdWNrZXROYW1lOiB7XG4gICAgICAgIFJlZjogXCJteWJ1Y2tldDE2MEY4MTMyXCJcbiAgICAgIH0sXG4gICAgICBMb2dGaWxlUHJlZml4OiBcImFjY2Vzcy1sb2dzXCJcbiAgICB9LFxuICAgIFB1YmxpY0FjY2Vzc0Jsb2NrQ29uZmlndXJhdGlvbjoge1xuICAgICAgQmxvY2tQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgQmxvY2tQdWJsaWNQb2xpY3k6IHRydWUsXG4gICAgICBJZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgUmVzdHJpY3RQdWJsaWNCdWNrZXRzOiB0cnVlXG4gICAgfVxuICB9KSk7XG59KTtcblxudGVzdCgnczMgYnVja2V0IHZlcnNpb25pbmcgdHVybmVkIG9mZicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWlsZFMzQnVja2V0UmVzcG9uc2UgPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7XG4gICAgYnVja2V0UHJvcHM6IHtcbiAgICAgIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6ICdhY2Nlc3MtbG9ncycsXG4gICAgICB2ZXJzaW9uZWQ6IGZhbHNlXG4gICAgfVxuICB9KTtcblxuICBleHBlY3QoYnVpbGRTM0J1Y2tldFJlc3BvbnNlLmJ1Y2tldCkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KGJ1aWxkUzNCdWNrZXRSZXNwb25zZS5sb2dnaW5nQnVja2V0KS50b0JlRGVmaW5lZCgpO1xuXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQVdTOjpTMzo6QnVja2V0XCIsIHtcbiAgICBCdWNrZXRFbmNyeXB0aW9uOiB7XG4gICAgICBTZXJ2ZXJTaWRlRW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IFtcbiAgICAgICAge1xuICAgICAgICAgIFNlcnZlclNpZGVFbmNyeXB0aW9uQnlEZWZhdWx0OiB7XG4gICAgICAgICAgICBTU0VBbGdvcml0aG06IFwiQUVTMjU2XCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIF1cbiAgICB9LFxuICAgIExvZ2dpbmdDb25maWd1cmF0aW9uOiB7XG4gICAgICBEZXN0aW5hdGlvbkJ1Y2tldE5hbWU6IHtcbiAgICAgICAgUmVmOiBcIlMzTG9nZ2luZ0J1Y2tldDgwMEEyQjI3XCJcbiAgICAgIH0sXG4gICAgICBMb2dGaWxlUHJlZml4OiBcImFjY2Vzcy1sb2dzXCJcbiAgICB9LFxuICAgIFB1YmxpY0FjY2Vzc0Jsb2NrQ29uZmlndXJhdGlvbjoge1xuICAgICAgQmxvY2tQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgQmxvY2tQdWJsaWNQb2xpY3k6IHRydWUsXG4gICAgICBJZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgUmVzdHJpY3RQdWJsaWNCdWNrZXRzOiB0cnVlXG4gICAgfVxuICB9KSk7XG59KTtcblxudGVzdCgnczMgYnVja2V0IHdpdGggTG9nZ2luZ0J1Y2tldCBhbmQgYXV0byBkZWxldGUgb2JqZWN0cycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBidWlsZFMzQnVja2V0UmVzcG9uc2UgPSBkZWZhdWx0cy5idWlsZFMzQnVja2V0KHN0YWNrLCB7XG4gICAgbG9nZ2luZ0J1Y2tldFByb3BzOiB7XG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZVxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KGJ1aWxkUzNCdWNrZXRSZXNwb25zZS5idWNrZXQpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UubG9nZ2luZ0J1Y2tldCkudG9CZURlZmluZWQoKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6UzM6OkJ1Y2tldFwiLCB7XG4gICAgQWNjZXNzQ29udHJvbDogXCJMb2dEZWxpdmVyeVdyaXRlXCJcbiAgfSkpO1xuXG4gIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKFwiQ3VzdG9tOjpTM0F1dG9EZWxldGVPYmplY3RzXCIsIHtcbiAgICBTZXJ2aWNlVG9rZW46IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiQ3VzdG9tUzNBdXRvRGVsZXRlT2JqZWN0c0N1c3RvbVJlc291cmNlUHJvdmlkZXJIYW5kbGVyOUQ5MDE4NEZcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH0sXG4gICAgQnVja2V0TmFtZToge1xuICAgICAgUmVmOiBcIlMzTG9nZ2luZ0J1Y2tldDgwMEEyQjI3XCJcbiAgICB9XG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdzMyBidWNrZXQgdmVyc2lvbmluZyB0dXJuZWQgb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgYnVpbGRTM0J1Y2tldFJlc3BvbnNlID0gZGVmYXVsdHMuYnVpbGRTM0J1Y2tldChzdGFjaywge1xuICAgIGJ1Y2tldFByb3BzOiB7XG4gICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OiAnYWNjZXNzLWxvZ3MnLFxuICAgIH1cbiAgfSk7XG5cbiAgZXhwZWN0KGJ1aWxkUzNCdWNrZXRSZXNwb25zZS5idWNrZXQpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChidWlsZFMzQnVja2V0UmVzcG9uc2UubG9nZ2luZ0J1Y2tldCkudG9CZURlZmluZWQoKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6UzM6OkJ1Y2tldFwiLCB7XG4gICAgQnVja2V0RW5jcnlwdGlvbjoge1xuICAgICAgU2VydmVyU2lkZUVuY3J5cHRpb25Db25maWd1cmF0aW9uOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBTZXJ2ZXJTaWRlRW5jcnlwdGlvbkJ5RGVmYXVsdDoge1xuICAgICAgICAgICAgU1NFQWxnb3JpdGhtOiBcIkFFUzI1NlwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSxcbiAgICBMaWZlY3ljbGVDb25maWd1cmF0aW9uOiB7XG4gICAgICBSdWxlczogW1xuICAgICAgICB7XG4gICAgICAgICAgTm9uY3VycmVudFZlcnNpb25UcmFuc2l0aW9uczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBTdG9yYWdlQ2xhc3M6IFwiR0xBQ0lFUlwiLFxuICAgICAgICAgICAgICBUcmFuc2l0aW9uSW5EYXlzOiA5MFxuICAgICAgICAgICAgfVxuICAgICAgICAgIF0sXG4gICAgICAgICAgU3RhdHVzOiBcIkVuYWJsZWRcIlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSxcbiAgICBMb2dnaW5nQ29uZmlndXJhdGlvbjoge1xuICAgICAgRGVzdGluYXRpb25CdWNrZXROYW1lOiB7XG4gICAgICAgIFJlZjogXCJTM0xvZ2dpbmdCdWNrZXQ4MDBBMkIyN1wiXG4gICAgICB9LFxuICAgICAgTG9nRmlsZVByZWZpeDogXCJhY2Nlc3MtbG9nc1wiXG4gICAgfSxcbiAgICBQdWJsaWNBY2Nlc3NCbG9ja0NvbmZpZ3VyYXRpb246IHtcbiAgICAgIEJsb2NrUHVibGljQWNsczogdHJ1ZSxcbiAgICAgIEJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgSWdub3JlUHVibGljQWNsczogdHJ1ZSxcbiAgICAgIFJlc3RyaWN0UHVibGljQnVja2V0czogdHJ1ZVxuICAgIH0sXG4gICAgVmVyc2lvbmluZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgIFN0YXR1czogXCJFbmFibGVkXCJcbiAgICB9XG4gIH0pKTtcbn0pO1xuXG50ZXN0KCdTdXBwcmVzcyBjZm4tbmFnIHdhcm5pbmcgZm9yIHMzIGJ1Y2tldCBub3RpZmljYXRpb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gIGNvbnN0IGJ1aWxkUzNCdWNrZXRSZXNwb25zZSA9IGRlZmF1bHRzLmJ1aWxkUzNCdWNrZXQoc3RhY2ssIHt9KTtcbiAgY29uc3QgYnVpbGRRdWV1ZVJlc3BvbnNlID0gZGVmYXVsdHMuYnVpbGRRdWV1ZShzdGFjaywgXCJTM0J1Y2tldE5vdGlmaWNhdGlvblF1ZXVlXCIsIHt9KTtcbiAgYnVpbGRTM0J1Y2tldFJlc3BvbnNlLmJ1Y2tldC5hZGRFdmVudE5vdGlmaWNhdGlvbihzMy5FdmVudFR5cGUuT0JKRUNUX0NSRUFURUQsIG5ldyBzM24uU3FzRGVzdGluYXRpb24oYnVpbGRRdWV1ZVJlc3BvbnNlLnF1ZXVlKSk7XG4gIGRlZmF1bHRzLmFkZENmbk5hZ1MzQnVja2V0Tm90aWZpY2F0aW9uUnVsZXNUb1N1cHByZXNzKHN0YWNrLCBcIkJ1Y2tldE5vdGlmaWNhdGlvbnNIYW5kbGVyMDUwYTA1ODdiNzU0NDU0N2JmMzI1ZjA5NGEzZGI4MzRcIik7XG5cbiAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoXCJBV1M6OkxhbWJkYTo6RnVuY3Rpb25cIiwge1xuICAgIE1ldGFkYXRhOiB7XG4gICAgICBjZm5fbmFnOiB7XG4gICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFwiVzU4XCIsXG4gICAgICAgICAgICByZWFzb246IFwiTGFtYmRhIGZ1bmN0aW9ucyBoYXMgdGhlIHJlcXVpcmVkIHBlcm1pc3Npb24gdG8gd3JpdGUgQ2xvdWRXYXRjaCBMb2dzLiBJdCB1c2VzIGN1c3RvbSBwb2xpY3kgaW5zdGVhZCBvZiBhcm46YXdzOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIHdpdGggdGlnaHRlciBwZXJtaXNzaW9ucy5cIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6ICdXODknLFxuICAgICAgICAgICAgcmVhc29uOiBgVGhpcyBpcyBub3QgYSBydWxlIGZvciB0aGUgZ2VuZXJhbCBjYXNlLCBqdXN0IGZvciBzcGVjaWZpYyB1c2UgY2FzZXMvaW5kdXN0cmllc2BcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGlkOiAnVzkyJyxcbiAgICAgICAgICAgIHJlYXNvbjogYEltcG9zc2libGUgZm9yIHVzIHRvIGRlZmluZSB0aGUgY29ycmVjdCBjb25jdXJyZW5jeSBmb3IgY2xpZW50c2BcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH1cbiAgICB9XG4gIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcblxuICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZShcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIE1ldGFkYXRhOiB7XG4gICAgICBjZm5fbmFnOiB7XG4gICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgaWQ6IFwiVzEyXCIsXG4gICAgICAgICAgICByZWFzb246IFwiQnVja2V0IHJlc291cmNlIGlzICcqJyBkdWUgdG8gY2lyY3VsYXIgZGVwZW5kZW5jeSB3aXRoIGJ1Y2tldCBhbmQgcm9sZSBjcmVhdGlvbiBhdCB0aGUgc2FtZSB0aW1lXCJcbiAgICAgICAgICB9XG4gICAgICAgIF1cbiAgICAgIH1cbiAgICB9XG4gIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcbn0pO1xuXG50ZXN0KCd0ZXN0IHMzQnVja2V0IHJlbW92YWxQb2xpY3kgb3ZlcnJpZGUnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgZGVmYXVsdHMuYnVpbGRTM0J1Y2tldChzdGFjaywge1xuICAgIGJ1Y2tldFByb3BzOiB7XG4gICAgICByZW1vdmFsUG9saWN5OiBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSxcbiAgfSwgJ3Rlc3QtYnVja2V0Jyk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OlMzOjpCdWNrZXRcIiwge1xuICAgIFR5cGU6ICdBV1M6OlMzOjpCdWNrZXQnLFxuICAgIFByb3BlcnRpZXM6IHtcbiAgICAgIEFjY2Vzc0NvbnRyb2w6IFwiTG9nRGVsaXZlcnlXcml0ZVwiXG4gICAgfSxcbiAgICBVcGRhdGVSZXBsYWNlUG9saWN5OiBcIkRlbGV0ZVwiLFxuICAgIERlbGV0aW9uUG9saWN5OiBcIkRlbGV0ZVwiXG4gIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pO1xufSk7XG5cbnRlc3QoJ3MzIGJ1Y2tldCB3aXRoIGxvZ2dpbmcgdHVybmVkIG9mZicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCByZXNwYnVpbGRTM0J1Y2tldFJlc3BvbnNlbnNlID0gZGVmYXVsdHMuYnVpbGRTM0J1Y2tldChzdGFjaywge1xuICAgIGxvZ1MzQWNjZXNzTG9nczogZmFsc2UsXG4gIH0pO1xuXG4gIGV4cGVjdChyZXNwYnVpbGRTM0J1Y2tldFJlc3BvbnNlbnNlLmJ1Y2tldCkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KHJlc3BidWlsZFMzQnVja2V0UmVzcG9uc2Vuc2UubG9nZ2luZ0J1Y2tldCkubm90LnRvQmVEZWZpbmVkKCk7XG5cbiAgZXhwZWN0KHN0YWNrKS5ub3QudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpTMzo6QnVja2V0XCIsIHtcbiAgICBMb2dnaW5nQ29uZmlndXJhdGlvbjoge1xuICAgIH0sXG4gIH0pO1xufSk7XG4iXX0=