"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const kms = require("aws-cdk-lib/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test('test s3Bucket override versioningConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        versioned: false
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-verioning', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    });
});
test('test s3Bucket override bucketEncryption', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        encryption: s3.BucketEncryption.KMS,
        encryptionKey: new kms.Key(stack, 'mykeyid')
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-encryption', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            "Fn::GetAtt": [
                                "mykeyidFA4203B0",
                                "Arn"
                            ]
                        },
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        },
    });
});
test('test s3Bucket override publicAccessBlockConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        blockPublicAccess: s3.BlockPublicAccess.BLOCK_ACLS
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-publicAccessBlock', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            IgnorePublicAcls: true
        },
    });
});
test('test s3Bucket add lifecycleConfiguration', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        lifecycleRules: [{
                expiration: aws_cdk_lib_1.Duration.days(365)
            }]
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-lifecycle', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    Status: 'Enabled',
                    ExpirationInDays: 365,
                }
            ]
        }
    });
});
test('test s3Bucket override serverAccessLogsBucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const myLoggingBucket = new s3.Bucket(stack, 'MyS3LoggingBucket', defaults.DefaultS3Props());
    const myS3Props = defaults.DefaultS3Props(myLoggingBucket);
    defaults.buildS3Bucket(stack, {
        bucketProps: myS3Props
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "MyS3LoggingBucket119BE896"
            }
        }
    });
});
test('test createAlbLoggingBucket()', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.createAlbLoggingBucket(stack, 'test-bucket', {
        bucketName: 'test-name'
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketName: 'test-name'
    });
});
test('Test bucket policy that only accepts SSL requests only', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            enforceSSL: true
        }
    }, 'test-bucket');
    expect(stack).toHaveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testbucketS3Bucket87F6BFFC",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "testbucketS3Bucket87F6BFFC",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('Test bucket policy that accepts any requests', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            enforceSSL: false
        }
    }, 'test-bucket');
    expect(stack).not.toHaveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testbucketS3Bucket87F6BFFC",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "testbucketS3Bucket87F6BFFC",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('Test enforcing SSL when bucketProps is not provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {}, 'test-bucket');
    expect(stack).toHaveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testbucketS3Bucket87F6BFFC",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "testbucketS3Bucket87F6BFFC",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('Test enforcing SSL when bucketProps is provided and enforceSSL is not set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            versioned: false,
            publicReadAccess: false
        }
    }, 'test-bucket');
    expect(stack).toHaveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testbucketS3Bucket87F6BFFC",
                                "Arn"
                            ]
                        },
                        {
                            "Fn::Join": [
                                "",
                                [
                                    {
                                        "Fn::GetAtt": [
                                            "testbucketS3Bucket87F6BFFC",
                                            "Arn"
                                        ]
                                    },
                                    "/*"
                                ]
                            ]
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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