"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const ec2 = require("aws-cdk-lib/aws-ec2");
const defaults = require("../");
require("@aws-cdk/assert/jest");
test('Test deployment with VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    const buildSagemakerNotebookResponse = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
    });
    // Assertion
    expect(buildSagemakerNotebookResponse.vpc?.privateSubnets.length).toEqual(2);
    expect(buildSagemakerNotebookResponse.vpc?.publicSubnets.length).toEqual(2);
    expect(buildSagemakerNotebookResponse.notebook.instanceType).toEqual('ml.t2.medium');
    expect(buildSagemakerNotebookResponse.securityGroup).toBeInstanceOf(ec2.SecurityGroup);
});
test('Test deployment without VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    const buildSagemakerNotebookResponse = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: false,
    });
    // Assertion
    expect(buildSagemakerNotebookResponse.vpc).not.toBeDefined();
    expect(buildSagemakerNotebookResponse.notebook).toBeDefined();
    expect(buildSagemakerNotebookResponse.securityGroup).not.toBeDefined();
});
test('Test deployment w/ existing VPC', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    // Build Sagemaker Notebook Instance
    const buildSagemakerNotebookResponse = defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef'],
        },
    });
    expect(buildSagemakerNotebookResponse.notebook).toBeDefined();
    expect(buildSagemakerNotebookResponse.vpc).not.toBeDefined();
    expect(buildSagemakerNotebookResponse.securityGroup).not.toBeDefined();
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        SecurityGroupIds: ['sg-deadbeef'],
        SubnetId: 'subnet-deadbeef',
    });
});
test('Test deployment w/ override', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    const key = new kms.Key(stack, 'MyEncryptionKey');
    // Build Sagemaker Notebook Instance
    defaults.buildSagemakerNotebook(stack, {
        role: sagemakerRole,
        sagemakerNotebookProps: {
            instanceType: 'ml.c4.2xlarge',
            kmsKeyId: key.keyArn,
        },
    });
    expect(stack).toHaveResource('AWS::SageMaker::NotebookInstance', {
        DirectInternetAccess: 'Disabled',
        InstanceType: 'ml.c4.2xlarge',
        KmsKeyId: {
            'Fn::GetAtt': ['MyEncryptionKeyD795679F', 'Arn'],
        },
    });
});
test('Test exception', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const sagemakerRole = new iam.Role(stack, 'SagemakerRole', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
    });
    expect(() => {
        // Build Sagemaker Notebook Instance
        defaults.buildSagemakerNotebook(stack, {
            role: sagemakerRole,
            deployInsideVpc: true,
            sagemakerNotebookProps: {
                subnetId: 'subnet-deadbeef',
            },
        });
    }).toThrowError();
});
test('Test exception for not providing primaryContainer in modelProps', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, {
            modelProps: {},
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
test('Test exception for not providing modelProps', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
test('Test exception for not providing modelProps or existingSagemkaerObj', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.BuildSagemakerEndpoint(stack, { vpc });
    };
    // Assertion 1
    expect(app).toThrowError();
});
test('Test exception for not providing private or isolated subnets in an existing vpc', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // create a VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        userVpcProps: {
            natGateways: 0,
            subnetConfiguration: [
                {
                    cidrMask: 18,
                    name: 'public',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
            ],
        },
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Build Sagemaker Inference Endpoint
        defaults.deploySagemakerEndpoint(stack, {
            modelProps: {
                primaryContainer: {
                    image: '<AccountId>.dkr.ecr.<region>.amazonaws.com/linear-learner:latest',
                    modelDataUrl: 's3://<bucket-name>/<prefix>/model.tar.gz',
                },
            },
            vpc,
        });
    };
    // Assertion 1
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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