"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const ec2 = require("aws-cdk-lib/aws-ec2");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "0.0.0.0/0",
                Description: "Allow all outbound traffic by default",
                IpProtocol: "-1",
            },
        ],
    });
});
test("Test deployment with ingress rules", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) }], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupIngress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
        ],
    });
});
test("Test deployment with egress rule", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: false,
    }, [], [
        { peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) },
        { peer: ec2.Peer.ipv4("2.2.2.2/24"), connection: ec2.Port.tcp(200) },
    ]);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
            {
                CidrIp: "2.2.2.2/24",
                Description: "from 2.2.2.2/24:200",
                FromPort: 200,
                IpProtocol: "tcp",
                ToPort: 200,
            },
        ],
    });
});
test("Test self referencing security group", () => {
    const testPort = 33333;
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.CreateSelfReferencingSecurityGroup(stack, "testsg", vpc, testPort);
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: testPort,
        ToPort: testPort,
        GroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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