"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const sqs_helper_1 = require("../lib/sqs-helper");
const kms = require("aws-cdk-lib/aws-kms");
const __1 = require("../");
test('Test deployment w/ encryptionMasterKey set on queueProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = new kms.Key(stack, 'EncryptionKey', {
        description: 'kms-key-description'
    });
    defaults.buildQueue(stack, 'queue', {
        queueProps: {
            encryptionMasterKey: cmk
        }
    });
    __1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SQS::Queue', 'kms-key-description');
});
test('Test deployment w/ imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: defaults.buildEncryptionKey(stack)
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
test('Test deployment without imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue",
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test('Test deployment w/ construct created encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const buildQueueResponse = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
    expect(buildQueueResponse.queue).toBeDefined();
    expect(buildQueueResponse.key).toBeDefined();
});
test('Test DLQ when existing Queue Provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingQueue = new sqs.Queue(stack, 'test-queue');
    const buildDlqProps = {
        existingQueueObj: existingQueue,
    };
    const returnedQueue = defaults.buildDeadLetterQueue(stack, buildDlqProps);
    expect(returnedQueue).toBeUndefined();
    expect(stack).toCountResources("AWS::SQS::Queue", 1);
});
test('Test DLQ with all defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    sqs_helper_1.buildDeadLetterQueue(stack, {});
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test("Test DLQ with a provided properties", () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testQueueName = "test-unique252";
    const returnedQueue = sqs_helper_1.buildDeadLetterQueue(stack, {
        deadLetterQueueProps: {
            queueName: testQueueName,
        },
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(returnedQueue).toBeDefined();
});
test('Test DLQ with a provided maxReceiveCount', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testMaxReceiveCount = 31;
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {
        maxReceiveCount: testMaxReceiveCount
    });
    expect(dlqInterface?.maxReceiveCount).toEqual(testMaxReceiveCount);
});
test('Test returning an existing Queue', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testQueueName = 'existing-queue';
    const existingQueue = new sqs.Queue(stack, 'test-queue', {
        queueName: testQueueName
    });
    const buildQueueResponse = defaults.buildQueue(stack, 'newQueue', {
        existingQueueObj: existingQueue
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(existingQueue.queueName).toEqual(buildQueueResponse.queue.queueName);
    expect(buildQueueResponse.key).not.toBeDefined();
});
test('Test creating a queue with a DLQ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {});
    const buildQueueResponse = sqs_helper_1.buildQueue(stack, 'new-queue', {
        deadLetterQueue: dlqInterface
    });
    expect(stack).toCountResources("AWS::SQS::Queue", 2);
    expect(buildQueueResponse.queue).toBeDefined();
    expect(buildQueueResponse.queue.deadLetterQueue).toBeDefined();
});
test('Test creating a FIFO queue', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildQueueResponse = sqs_helper_1.buildQueue(stack, 'new-queue', {
        queueProps: {
            fifo: true
        }
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        FifoQueue: true
    });
    expect(buildQueueResponse.queue.fifo).toBe(true);
});
//# sourceMappingURL=data:application/json;base64,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