"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const cloudwatch_log_group_helper_1 = require("../lib/cloudwatch-log-group-helper");
const iam = require("aws-cdk-lib/aws-iam");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        definition: startState,
        stateMachineName: 'myStateMachine'
    });
    // Assertion
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(stack).toCountResources("AWS::Logs::LogGroup", 1);
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        StateMachineName: "myStateMachine"
    });
});
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Log group
    // const logGroup = new LogGroup(stack, 'myLogGroup', defaults.buildLogGroup(stack));
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(stack, 'StateMachineLogGroup');
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        definition: startState,
        logs: {
            destination: logGroup,
            level: sfn.LogLevel.FATAL
        }
    });
    // Assertion
    expect(stack).toCountResources("AWS::Logs::LogGroup", 1);
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        LoggingConfiguration: {
            Destinations: [{
                    CloudWatchLogsLogGroup: {
                        LogGroupArn: {
                            "Fn::GetAtt": [
                                "StateMachineLogGroup15B91BCB",
                                "Arn"
                            ]
                        }
                    }
                }],
            Level: 'FATAL'
        }
    });
});
test('Check default Cloudwatch permissions', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "logs:CreateLogDelivery",
                        "logs:GetLogDelivery",
                        "logs:UpdateLogDelivery",
                        "logs:DeleteLogDelivery",
                        "logs:ListLogDeliveries"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "logs:PutResourcePolicy",
                        "logs:DescribeResourcePolicies",
                        "logs:DescribeLogGroups"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":logs:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('Count State Machine CW Alarms', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        definition: startState
    });
    const cwList = defaults.buildStepFunctionCWAlarms(stack, buildStateMachineResponse.stateMachine);
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(cwList.length).toEqual(3);
});
test('Test deployment with custom role', () => {
    const descriptionText = 'Custom role for State Machine';
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    const customRole = new iam.Role(stack, 'custom-role', {
        assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        description: descriptionText,
        inlinePolicies: {
            InvokePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:s3:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
                        actions: ['s3:ListBucket']
                    })]
            })
        }
    });
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        definition: startState,
        role: customRole
    });
    // Assertion
    expect(stack).toCountResources("AWS::IAM::Role", 1);
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(stack).toHaveResource("AWS::IAM::Role", {
        Description: descriptionText
    });
});
test('Confirm format of name', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const buildStateMachineResponse = defaults.buildStateMachine(stack, {
        stateMachineName: 'myStateMachine',
        definition: startState,
    });
    // Assertion
    expect(buildStateMachineResponse.stateMachine).toBeDefined();
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        StateMachineName: "myStateMachine"
    });
    // Perform some fancy stuff to examine the specifics of the LogGroupName
    const expectedPrefix = '/aws/vendedlogs/states/constructs/';
    const lengthOfDatetimeSuffix = 13;
    const LogGroup = assertions_1.Template.fromStack(stack).findResources("AWS::Logs::LogGroup");
    const logName = LogGroup.StateMachineLogGroup15B91BCB.Properties.LogGroupName;
    const suffix = logName.slice(-lengthOfDatetimeSuffix);
    // Look for the expected Prefix and the 13 digit time suffix
    expect(logName.slice(0, expectedPrefix.length)).toEqual(expectedPrefix);
    expect(IsWholeNumber(suffix)).not.toBe(false);
});
function IsWholeNumber(target) {
    const numberPattern = /[0-9]{13}/;
    return target.match(numberPattern) !== null;
}
//# sourceMappingURL=data:application/json;base64,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