"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.expectKmsKeyAttachedToCorrectResource = exports.CreateTestCache = exports.suppressAutoDeleteHandlerWarnings = exports.getFakeCertificate = exports.getTestVpc = exports.generateIntegStackName = exports.CreateScrapBucket = exports.fakeEcrRepoArn = void 0;
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
const utils_1 = require("../lib/utils");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const path = require("path");
const cache = require("aws-cdk-lib/aws-elasticache");
const ec2 = require("aws-cdk-lib/aws-ec2");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
const assertions_1 = require("aws-cdk-lib/assertions");
exports.fakeEcrRepoArn = 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo';
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
// Helper Functions
function getTestVpc(stack, publicFacing = true, userVpcProps) {
    return vpc_helper_1.buildVpc(stack, {
        defaultVpcProps: publicFacing ?
            vpc_defaults_1.DefaultPublicPrivateVpcProps() :
            vpc_defaults_1.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            ipAddresses: ec2.IpAddresses.cidr('172.168.0.0/16'),
        },
        userVpcProps
    });
}
exports.getTestVpc = getTestVpc;
function getFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
exports.getFakeCertificate = getFakeCertificate;
function suppressAutoDeleteHandlerWarnings(stack) {
    aws_cdk_lib_1.Stack.of(stack).node.children.forEach(child => {
        if (child.node.id === 'Custom::S3AutoDeleteObjectsCustomResourceProvider') {
            const handlerFunction = child.node.findChild('Handler');
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W58", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W89", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W92", reason: "CDK generated custom resource" }]);
        }
    });
}
exports.suppressAutoDeleteHandlerWarnings = suppressAutoDeleteHandlerWarnings;
function CreateTestCache(scope, id, vpc, port) {
    const cachePort = port ?? elasticache_defaults_1.GetDefaultCachePort();
    // Create the subnet group from all the isolated subnets in the VPC
    const subnetGroup = elasticache_helper_1.createCacheSubnetGroup(scope, vpc, id);
    const emptySG = new ec2.SecurityGroup(scope, `${id}-cachesg`, {
        vpc,
        allowAllOutbound: true,
    });
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W40", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W5", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W36", reason: "Test Resource" }]);
    const cacheProps = {
        clusterName: `${id}-cdk-cluster`,
        cacheNodeType: "cache.t3.medium",
        engine: "memcached",
        numCacheNodes: 2,
        port: cachePort,
        azMode: "cross-az",
        vpcSecurityGroupIds: [emptySG.securityGroupId],
        cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
    };
    const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, cacheProps);
    newCache.addDependency(subnetGroup);
    return newCache;
}
exports.CreateTestCache = CreateTestCache;
/**
 * Asserts that a KMS key with a specific description exists on a resource
 *
 * @param stack The CloudFormation Stack that contains the to validate.
 * @param parentResourceType The type of CloudFormation Resource that should have the key set on it, e.g., `AWS::SNS::Topic`, etc...
 * @param description The value of the Description property on the KMS Key
 */
function expectKmsKeyAttachedToCorrectResource(stack, parentResourceType, keyDescription) {
    const template = assertions_1.Template.fromStack(stack);
    const resource = template.findResources('AWS::KMS::Key', {
        Properties: {
            Description: assertions_1.Match.exact(keyDescription)
        }
    });
    const [logicalId] = Object.keys(resource);
    expect(stack).toHaveResource(parentResourceType, {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                logicalId,
                "Arn"
            ]
        }
    });
}
exports.expectKmsKeyAttachedToCorrectResource = expectKmsKeyAttachedToCorrectResource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC1oZWxwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LWhlbHBlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUlILCtDQUEyRTtBQUMzRSw2Q0FBbUQ7QUFDbkQsa0RBQTZDO0FBQzdDLHNEQUE0RjtBQUM1Rix3Q0FBa0U7QUFDbEUsa0VBQW9FO0FBQ3BFLDZCQUE2QjtBQUM3QixxREFBcUQ7QUFDckQsMkNBQTJDO0FBQzNDLDBEQUEwRDtBQUUxRCxzRUFBa0U7QUFDbEUsdURBQXlEO0FBRTVDLFFBQUEsY0FBYyxHQUFHLHlEQUF5RCxDQUFDO0FBRXhGLCtFQUErRTtBQUMvRSxTQUFnQixpQkFBaUIsQ0FBQyxLQUFnQixFQUFFLEtBQXlCO0lBQzNFLE1BQU0sWUFBWSxHQUFHO1FBQ25CLFNBQVMsRUFBRSxJQUFJO1FBQ2YsYUFBYSxFQUFFLDJCQUFhLENBQUMsT0FBTztRQUNwQyxVQUFVLEVBQUUseUJBQWdCLENBQUMsVUFBVTtLQUN4QyxDQUFDO0lBRUYsSUFBSSxnQkFBNkIsQ0FBQztJQUNsQyxJQUFJLEtBQUssRUFBRTtRQUNULGdCQUFnQixHQUFHLHFCQUFhLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxDQUFDO0tBQ3ZEO1NBQU07UUFDTCxnQkFBZ0IsR0FBRyxZQUFZLENBQUM7S0FDakM7SUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLGVBQU0sQ0FDN0IsS0FBSyxFQUNMLHdCQUF3QixFQUN4QixnQkFBZ0IsQ0FDakIsQ0FBQztJQUVGLDJCQUFtQixDQUFDLFlBQVksRUFBRTtRQUNoQztZQUNFLEVBQUUsRUFBRSxLQUFLO1lBQ1QsTUFBTSxFQUFFLGlJQUFpSTtTQUMxSTtRQUNEO1lBQ0UsRUFBRSxFQUFFLEtBQUs7WUFDVCxNQUFNLEVBQUUsaUlBQWlJO1NBQzFJO1FBQ0Q7WUFDRSxFQUFFLEVBQUUsS0FBSztZQUNULE1BQU0sRUFBRSxrSEFBa0g7U0FDM0g7S0FDRixDQUFDLENBQUM7SUFFSCxPQUFPLFlBQVksQ0FBQztBQUN0QixDQUFDO0FBcENELDhDQW9DQztBQUVEOzs7OztHQUtHO0FBQ0gsU0FBZ0Isc0JBQXNCLENBQUMsUUFBZ0I7SUFDckQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0lBQzdELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzNFLE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUM7QUFKRCx3REFJQztBQUVELG1CQUFtQjtBQUNuQixTQUFnQixVQUFVLENBQUMsS0FBWSxFQUFFLGVBQXdCLElBQUksRUFBRSxZQUEyQjtJQUNoRyxPQUFPLHFCQUFRLENBQUMsS0FBSyxFQUFFO1FBQ3JCLGVBQWUsRUFBRSxZQUFZLENBQUMsQ0FBQztZQUM3QiwyQ0FBNEIsRUFBRSxDQUFDLENBQUM7WUFDaEMsc0NBQXVCLEVBQUU7UUFDM0IsaUJBQWlCLEVBQUU7WUFDakIsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztTQUNwRDtRQUNELFlBQVk7S0FDYixDQUFDLENBQUM7QUFDTCxDQUFDO0FBWkQsZ0NBWUM7QUFFRCxTQUFnQixrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEVBQVU7SUFDN0QsT0FBTyxHQUFHLENBQUMsV0FBVyxDQUFDLGtCQUFrQixDQUN2QyxLQUFLLEVBQ0wsRUFBRSxFQUNGLHFGQUFxRixDQUN0RixDQUFDO0FBQ0osQ0FBQztBQU5ELGdEQU1DO0FBRUQsU0FBZ0IsaUNBQWlDLENBQUMsS0FBWTtJQUM1RCxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUM1QyxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsRUFBRSxLQUFLLG1EQUFtRCxFQUFFO1lBQ3pFLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBZ0IsQ0FBQztZQUN2RSwyQkFBbUIsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLCtCQUErQixFQUFDLENBQUMsQ0FBQyxDQUFDO1lBQzlGLDJCQUFtQixDQUFDLGVBQWUsRUFBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsK0JBQStCLEVBQUMsQ0FBQyxDQUFDLENBQUM7WUFDOUYsMkJBQW1CLENBQUMsZUFBZSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSwrQkFBK0IsRUFBQyxDQUFDLENBQUMsQ0FBQztTQUMvRjtJQUNILENBQUMsQ0FBQyxDQUFDO0FBRUwsQ0FBQztBQVZELDhFQVVDO0FBRUQsU0FBZ0IsZUFBZSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEdBQWEsRUFBRSxJQUFhO0lBQ3hGLE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSwwQ0FBbUIsRUFBRSxDQUFDO0lBRWhELG1FQUFtRTtJQUNuRSxNQUFNLFdBQVcsR0FBRywyQ0FBc0IsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQzNELE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsR0FBRyxFQUFFLFVBQVUsRUFBRTtRQUM1RCxHQUFHO1FBQ0gsZ0JBQWdCLEVBQUUsSUFBSTtLQUN2QixDQUFDLENBQUM7SUFDSCwyQkFBbUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN2RSwyQkFBbUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUN0RSwyQkFBbUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUV2RSxNQUFNLFVBQVUsR0FBRztRQUNqQixXQUFXLEVBQUUsR0FBRyxFQUFFLGNBQWM7UUFDaEMsYUFBYSxFQUFFLGlCQUFpQjtRQUNoQyxNQUFNLEVBQUUsV0FBVztRQUNuQixhQUFhLEVBQUUsQ0FBQztRQUNoQixJQUFJLEVBQUUsU0FBUztRQUNmLE1BQU0sRUFBRSxVQUFVO1FBQ2xCLG1CQUFtQixFQUFFLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQztRQUM5QyxvQkFBb0IsRUFBRSxXQUFXLENBQUMsb0JBQW9CO0tBQ3ZELENBQUM7SUFFRixNQUFNLFFBQVEsR0FBRyxJQUFJLEtBQUssQ0FBQyxlQUFlLENBQ3hDLEtBQUssRUFDTCxHQUFHLEVBQUUsVUFBVSxFQUNmLFVBQVUsQ0FDWCxDQUFDO0lBQ0YsUUFBUSxDQUFDLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUNwQyxPQUFPLFFBQVEsQ0FBQztBQUNsQixDQUFDO0FBL0JELDBDQStCQztBQUVEOzs7Ozs7R0FNRztBQUNILFNBQWdCLHFDQUFxQyxDQUFDLEtBQVksRUFBRSxrQkFBMEIsRUFBRSxjQUFzQjtJQUNwSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLGVBQWUsRUFBRTtRQUN2RCxVQUFVLEVBQUU7WUFDVixXQUFXLEVBQUUsa0JBQUssQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1NBQ3pDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFFLFNBQVMsQ0FBRSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDNUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtRQUMvQyxjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1osU0FBUztnQkFDVCxLQUFLO2FBQ047U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUM7QUFqQkQsc0ZBaUJDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBCdWNrZXQsIEJ1Y2tldFByb3BzLCBCdWNrZXRFbmNyeXB0aW9uIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1zM1wiO1xuaW1wb3J0IHsgUmVtb3ZhbFBvbGljeSwgU3RhY2sgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IGJ1aWxkVnBjIH0gZnJvbSAnLi4vbGliL3ZwYy1oZWxwZXInO1xuaW1wb3J0IHsgRGVmYXVsdFB1YmxpY1ByaXZhdGVWcGNQcm9wcywgRGVmYXVsdElzb2xhdGVkVnBjUHJvcHMgfSBmcm9tICcuLi9saWIvdnBjLWRlZmF1bHRzJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMsIGFkZENmblN1cHByZXNzUnVsZXMgfSBmcm9tIFwiLi4vbGliL3V0aWxzXCI7XG5pbXBvcnQgeyBjcmVhdGVDYWNoZVN1Ym5ldEdyb3VwICB9IGZyb20gXCIuLi9saWIvZWxhc3RpY2FjaGUtaGVscGVyXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgY2FjaGUgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVsYXN0aWNhY2hlJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGFjbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtY2VydGlmaWNhdGVtYW5hZ2VyJztcbmltcG9ydCB7IENmbkZ1bmN0aW9uIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IEdldERlZmF1bHRDYWNoZVBvcnQgfSBmcm9tIFwiLi4vbGliL2VsYXN0aWNhY2hlLWRlZmF1bHRzXCI7XG5pbXBvcnQgeyBNYXRjaCwgVGVtcGxhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hc3NlcnRpb25zJztcblxuZXhwb3J0IGNvbnN0IGZha2VFY3JSZXBvQXJuID0gJ2Fybjphd3M6ZWNyOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6cmVwb3NpdG9yeS9mYWtlLXJlcG8nO1xuXG4vLyBDcmVhdGVzIGEgYnVja2V0IHVzZWQgZm9yIHRlc3RpbmcgLSBtaW5pbWFsIHByb3BlcnRpZXMsIGRlc3Ryb3llZCBhZnRlciB0ZXN0XG5leHBvcnQgZnVuY3Rpb24gQ3JlYXRlU2NyYXBCdWNrZXQoc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM/OiBCdWNrZXRQcm9wcyB8IGFueSkge1xuICBjb25zdCBkZWZhdWx0UHJvcHMgPSB7XG4gICAgdmVyc2lvbmVkOiB0cnVlLFxuICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQsXG4gIH07XG5cbiAgbGV0IHN5bnRoZXNpemVkUHJvcHM6IEJ1Y2tldFByb3BzO1xuICBpZiAocHJvcHMpIHtcbiAgICBzeW50aGVzaXplZFByb3BzID0gb3ZlcnJpZGVQcm9wcyhkZWZhdWx0UHJvcHMsIHByb3BzKTtcbiAgfSBlbHNlIHtcbiAgICBzeW50aGVzaXplZFByb3BzID0gZGVmYXVsdFByb3BzO1xuICB9XG5cbiAgY29uc3Qgc2NyaXB0QnVja2V0ID0gbmV3IEJ1Y2tldChcbiAgICBzY29wZSxcbiAgICBcImV4aXN0aW5nU2NyaXB0TG9jYXRpb25cIixcbiAgICBzeW50aGVzaXplZFByb3BzXG4gICk7XG5cbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhzY3JpcHRCdWNrZXQsIFtcbiAgICB7XG4gICAgICBpZDogXCJXNTFcIixcbiAgICAgIHJlYXNvbjogXCJUaGlzIFMzIGJ1Y2tldCBpcyBjcmVhdGVkIGZvciB1bml0LyBpbnRlZ3JhdGlvbiB0ZXN0aW5nIHB1cnBvc2VzIG9ubHkgYW5kIG5vdCBwYXJ0IG9mICAgICAgIHRoZSBhY3R1YWwgY29uc3RydWN0IGltcGxlbWVudGF0aW9uXCIsXG4gICAgfSxcbiAgICB7XG4gICAgICBpZDogXCJXMzVcIixcbiAgICAgIHJlYXNvbjogXCJUaGlzIFMzIGJ1Y2tldCBpcyBjcmVhdGVkIGZvciB1bml0LyBpbnRlZ3JhdGlvbiB0ZXN0aW5nIHB1cnBvc2VzIG9ubHkgYW5kIG5vdCBwYXJ0IG9mICAgICAgIHRoZSBhY3R1YWwgY29uc3RydWN0IGltcGxlbWVudGF0aW9uXCIsXG4gICAgfSxcbiAgICB7XG4gICAgICBpZDogXCJXNDFcIixcbiAgICAgIHJlYXNvbjogXCJUaGlzIFMzIGJ1Y2tldCBpcyBjcmVhdGVkIGZvciB1bml0LyBpbnRlZ3JhdGlvbiB0ZXN0aW5nIHB1cnBvc2VzIG9ubHkgYW5kIG5vdCBwYXJ0IG9mICAgICAgIHRoZSBhY3R1YWwgY29uc3RydWN0XCIsXG4gICAgfVxuICBdKTtcblxuICByZXR1cm4gc2NyaXB0QnVja2V0O1xufVxuXG4vKipcbiAqIEBzdW1tYXJ5IENyZWF0ZXMgYSBzdGFjayBuYW1lIGZvciBJbnRlZ3JhdGlvbiB0ZXN0c1xuICogQHBhcmFtIHtzdHJpbmd9IGZpbGVuYW1lIC0gdGhlIGZpbGVuYW1lIG9mIHRoZSBpbnRlZyB0ZXN0XG4gKiBAcmV0dXJucyB7c3RyaW5nfSAtIGEgc3RyaW5nIHdpdGggY3VycmVudCBmaWxlbmFtZSBhZnRlciByZW1vdmluZyBhbnl0aGluZyBiZWZvcmUgdGhlIHByZWZpeCAnLicgYW5kIHN1ZmZpeCAnLmpzJ1xuICogZS5nLiAnaW50ZWcuYXBpZ2F0ZXdheS1keW5hbW9kYi1DUlVELmpzJyB3aWxsIHJldHVybiAnYXBpZ2F0ZXdheS1keW5hbW9kYi1DUlVEJ1xuICovXG5leHBvcnQgZnVuY3Rpb24gZ2VuZXJhdGVJbnRlZ1N0YWNrTmFtZShmaWxlbmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3QgZmlsZSA9IHBhdGguYmFzZW5hbWUoZmlsZW5hbWUsIHBhdGguZXh0bmFtZShmaWxlbmFtZSkpO1xuICBjb25zdCBzdGFja25hbWUgPSBmaWxlLnNsaWNlKGZpbGUubGFzdEluZGV4T2YoJy4nKSArIDEpLnJlcGxhY2UoL18vZywgJy0nKTtcbiAgcmV0dXJuIHN0YWNrbmFtZTtcbn1cblxuLy8gSGVscGVyIEZ1bmN0aW9uc1xuZXhwb3J0IGZ1bmN0aW9uIGdldFRlc3RWcGMoc3RhY2s6IFN0YWNrLCBwdWJsaWNGYWNpbmc6IGJvb2xlYW4gPSB0cnVlLCB1c2VyVnBjUHJvcHM/OiBlYzIuVnBjUHJvcHMpIHtcbiAgcmV0dXJuIGJ1aWxkVnBjKHN0YWNrLCB7XG4gICAgZGVmYXVsdFZwY1Byb3BzOiBwdWJsaWNGYWNpbmcgP1xuICAgICAgRGVmYXVsdFB1YmxpY1ByaXZhdGVWcGNQcm9wcygpIDpcbiAgICAgIERlZmF1bHRJc29sYXRlZFZwY1Byb3BzKCksXG4gICAgY29uc3RydWN0VnBjUHJvcHM6IHtcbiAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IHRydWUsXG4gICAgICBpcEFkZHJlc3NlczogZWMyLklwQWRkcmVzc2VzLmNpZHIoJzE3Mi4xNjguMC4wLzE2JyksXG4gICAgfSxcbiAgICB1c2VyVnBjUHJvcHNcbiAgfSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRGYWtlQ2VydGlmaWNhdGUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZyk6IGFjbS5JQ2VydGlmaWNhdGUge1xuICByZXR1cm4gYWNtLkNlcnRpZmljYXRlLmZyb21DZXJ0aWZpY2F0ZUFybihcbiAgICBzY29wZSxcbiAgICBpZCxcbiAgICBcImFybjphd3M6YWNtOnVzLWVhc3QtMToxMjM0NTY3ODkwMTI6Y2VydGlmaWNhdGUvMTExMTIyMjItMzMzMy0xMjM0LTEyMzQtMTIzNDU2Nzg5MDEyXCJcbiAgKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHN1cHByZXNzQXV0b0RlbGV0ZUhhbmRsZXJXYXJuaW5ncyhzdGFjazogU3RhY2spIHtcbiAgU3RhY2sub2Yoc3RhY2spLm5vZGUuY2hpbGRyZW4uZm9yRWFjaChjaGlsZCA9PiB7XG4gICAgaWYgKGNoaWxkLm5vZGUuaWQgPT09ICdDdXN0b206OlMzQXV0b0RlbGV0ZU9iamVjdHNDdXN0b21SZXNvdXJjZVByb3ZpZGVyJykge1xuICAgICAgY29uc3QgaGFuZGxlckZ1bmN0aW9uID0gY2hpbGQubm9kZS5maW5kQ2hpbGQoJ0hhbmRsZXInKSBhcyBDZm5GdW5jdGlvbjtcbiAgICAgIGFkZENmblN1cHByZXNzUnVsZXMoaGFuZGxlckZ1bmN0aW9uLCBbeyBpZDogXCJXNThcIiwgcmVhc29uOiBcIkNESyBnZW5lcmF0ZWQgY3VzdG9tIHJlc291cmNlXCJ9XSk7XG4gICAgICBhZGRDZm5TdXBwcmVzc1J1bGVzKGhhbmRsZXJGdW5jdGlvbiwgW3sgaWQ6IFwiVzg5XCIsIHJlYXNvbjogXCJDREsgZ2VuZXJhdGVkIGN1c3RvbSByZXNvdXJjZVwifV0pO1xuICAgICAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhoYW5kbGVyRnVuY3Rpb24sIFt7IGlkOiBcIlc5MlwiLCByZWFzb246IFwiQ0RLIGdlbmVyYXRlZCBjdXN0b20gcmVzb3VyY2VcIn1dKTtcbiAgICB9XG4gIH0pO1xuXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBDcmVhdGVUZXN0Q2FjaGUoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdnBjOiBlYzIuSVZwYywgcG9ydD86IG51bWJlcikge1xuICBjb25zdCBjYWNoZVBvcnQgPSBwb3J0ID8/IEdldERlZmF1bHRDYWNoZVBvcnQoKTtcblxuICAvLyBDcmVhdGUgdGhlIHN1Ym5ldCBncm91cCBmcm9tIGFsbCB0aGUgaXNvbGF0ZWQgc3VibmV0cyBpbiB0aGUgVlBDXG4gIGNvbnN0IHN1Ym5ldEdyb3VwID0gY3JlYXRlQ2FjaGVTdWJuZXRHcm91cChzY29wZSwgdnBjLCBpZCk7XG4gIGNvbnN0IGVtcHR5U0cgPSBuZXcgZWMyLlNlY3VyaXR5R3JvdXAoc2NvcGUsIGAke2lkfS1jYWNoZXNnYCwge1xuICAgIHZwYyxcbiAgICBhbGxvd0FsbE91dGJvdW5kOiB0cnVlLFxuICB9KTtcbiAgYWRkQ2ZuU3VwcHJlc3NSdWxlcyhlbXB0eVNHLCBbeyBpZDogXCJXNDBcIiwgcmVhc29uOiBcIlRlc3QgUmVzb3VyY2VcIiB9XSk7XG4gIGFkZENmblN1cHByZXNzUnVsZXMoZW1wdHlTRywgW3sgaWQ6IFwiVzVcIiwgcmVhc29uOiBcIlRlc3QgUmVzb3VyY2VcIiB9XSk7XG4gIGFkZENmblN1cHByZXNzUnVsZXMoZW1wdHlTRywgW3sgaWQ6IFwiVzM2XCIsIHJlYXNvbjogXCJUZXN0IFJlc291cmNlXCIgfV0pO1xuXG4gIGNvbnN0IGNhY2hlUHJvcHMgPSB7XG4gICAgY2x1c3Rlck5hbWU6IGAke2lkfS1jZGstY2x1c3RlcmAsXG4gICAgY2FjaGVOb2RlVHlwZTogXCJjYWNoZS50My5tZWRpdW1cIixcbiAgICBlbmdpbmU6IFwibWVtY2FjaGVkXCIsXG4gICAgbnVtQ2FjaGVOb2RlczogMixcbiAgICBwb3J0OiBjYWNoZVBvcnQsXG4gICAgYXpNb2RlOiBcImNyb3NzLWF6XCIsXG4gICAgdnBjU2VjdXJpdHlHcm91cElkczogW2VtcHR5U0cuc2VjdXJpdHlHcm91cElkXSxcbiAgICBjYWNoZVN1Ym5ldEdyb3VwTmFtZTogc3VibmV0R3JvdXAuY2FjaGVTdWJuZXRHcm91cE5hbWUsXG4gIH07XG5cbiAgY29uc3QgbmV3Q2FjaGUgPSBuZXcgY2FjaGUuQ2ZuQ2FjaGVDbHVzdGVyKFxuICAgIHNjb3BlLFxuICAgIGAke2lkfS1jbHVzdGVyYCxcbiAgICBjYWNoZVByb3BzXG4gICk7XG4gIG5ld0NhY2hlLmFkZERlcGVuZGVuY3koc3VibmV0R3JvdXApO1xuICByZXR1cm4gbmV3Q2FjaGU7XG59XG5cbi8qKlxuICogQXNzZXJ0cyB0aGF0IGEgS01TIGtleSB3aXRoIGEgc3BlY2lmaWMgZGVzY3JpcHRpb24gZXhpc3RzIG9uIGEgcmVzb3VyY2VcbiAqXG4gKiBAcGFyYW0gc3RhY2sgVGhlIENsb3VkRm9ybWF0aW9uIFN0YWNrIHRoYXQgY29udGFpbnMgdGhlIHRvIHZhbGlkYXRlLlxuICogQHBhcmFtIHBhcmVudFJlc291cmNlVHlwZSBUaGUgdHlwZSBvZiBDbG91ZEZvcm1hdGlvbiBSZXNvdXJjZSB0aGF0IHNob3VsZCBoYXZlIHRoZSBrZXkgc2V0IG9uIGl0LCBlLmcuLCBgQVdTOjpTTlM6OlRvcGljYCwgZXRjLi4uXG4gKiBAcGFyYW0gZGVzY3JpcHRpb24gVGhlIHZhbHVlIG9mIHRoZSBEZXNjcmlwdGlvbiBwcm9wZXJ0eSBvbiB0aGUgS01TIEtleVxuICovXG5leHBvcnQgZnVuY3Rpb24gZXhwZWN0S21zS2V5QXR0YWNoZWRUb0NvcnJlY3RSZXNvdXJjZShzdGFjazogU3RhY2ssIHBhcmVudFJlc291cmNlVHlwZTogc3RyaW5nLCBrZXlEZXNjcmlwdGlvbjogc3RyaW5nKSB7XG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgY29uc3QgcmVzb3VyY2UgPSB0ZW1wbGF0ZS5maW5kUmVzb3VyY2VzKCdBV1M6OktNUzo6S2V5Jywge1xuICAgIFByb3BlcnRpZXM6IHtcbiAgICAgIERlc2NyaXB0aW9uOiBNYXRjaC5leGFjdChrZXlEZXNjcmlwdGlvbilcbiAgICB9XG4gIH0pO1xuXG4gIGNvbnN0IFsgbG9naWNhbElkIF0gPSBPYmplY3Qua2V5cyhyZXNvdXJjZSk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UocGFyZW50UmVzb3VyY2VUeXBlLCB7XG4gICAgS21zTWFzdGVyS2V5SWQ6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIGxvZ2ljYWxJZCxcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgIH1cbiAgfSk7XG59XG4iXX0=