"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const defaults = require("../");
const cdk = require("aws-cdk-lib");
// Need 2 parts, but they can't overlap
// so we can explicitly find them in the results.
const parts = ['firstportionislong', 'secondsection'];
const nonAlphaParts = ['part-one', 'part-two'];
test('Test generateResourceName with a truncated part', () => {
    const result = defaults.generateResourceName(parts, 38);
    expect(result).toContain(parts[1]);
    expect(result).not.toContain(parts[0]);
    expect(result).toContain(parts[0].slice(0, 13));
});
test('Test generateResourceName with no truncated parts', () => {
    const result = defaults.generateResourceName(parts, 100);
    expect(result).toContain(parts[1]);
    expect(result).toContain(parts[0]);
    expect(result.length).toEqual(parts[0].length + parts[1].length + 12);
});
test('Test generateResourceName with non Aphanumeric', () => {
    const result = defaults.generateResourceName(nonAlphaParts, 100);
    expect(result).toContain('partoneparttwo');
});
test('Test generateResourceName with randomized extension', () => {
    const resultOne = defaults.generateResourceName(parts, 512, true);
    const startTime = (new Date()).getTime();
    // We need to ensure the time value appended changes between callls
    let currTime = startTime;
    while (currTime === startTime) {
        currTime = (new Date()).getTime();
    }
    const resultTwo = defaults.generateResourceName(parts, 512, true);
    expect(resultOne).toContain(parts[1]);
    expect(resultOne).toContain(parts[0]);
    expect(resultTwo).toContain(parts[1]);
    expect(resultTwo).toContain(parts[0]);
    expect(resultOne).not.toEqual(resultTwo);
    expect(resultOne.slice(0, -13)).toEqual(resultTwo.slice(0, -13));
});
test('Test generateIntegStackName', () => {
    const result = defaults.generateIntegStackName('integ.apigateway-dynamodb-CRUD.js');
    expect(result).toContain('apigateway-dynamodb-CRUD');
    const result1 = defaults.generateIntegStackName('integ.override_auth_api_keys.ts');
    expect(result1).toContain('override-auth-api-keys');
});
test('Test consolidate props with all args', () => {
    const arg1 = {
        val1: 11,
        val2: 12,
        val3: 13,
    };
    const arg2 = {
        val1: 21,
        val2: 22,
    };
    const arg3 = {
        val1: 31,
    };
    const result = defaults.consolidateProps(arg1, arg2, arg3);
    expect(result).toEqual({
        val1: 31,
        val2: 22,
        val3: 13,
    });
});
test('Test consolidate props with first and third args', () => {
    const arg1 = {
        val1: 11,
        val2: 12,
        val3: 13,
    };
    const arg3 = {
        val1: 31,
    };
    const result = defaults.consolidateProps(arg1, undefined, arg3);
    expect(result).toEqual({
        val1: 31,
        val2: 12,
        val3: 13,
    });
});
test('Test consolidate props with first and second args', () => {
    const arg1 = {
        val1: 11,
        val2: 12,
        val3: 13,
    };
    const arg2 = {
        val1: 21,
        val2: 22,
    };
    const result = defaults.consolidateProps(arg1, arg2);
    expect(result).toEqual({
        val1: 21,
        val2: 22,
        val3: 13,
    });
});
test('Test consolidate props with one arg', () => {
    const arg1 = {
        val1: 11,
        val2: 12,
        val3: 13,
    };
    const result = defaults.consolidateProps(arg1);
    expect(result).toEqual(arg1);
});
test('Test generateName sunny day for current construct with undefined name argument', () => {
    const stack = new cdk.Stack(undefined, "some-new-id");
    const newName = defaults.generateName(stack);
    // 5 is not specific, just checking the name is several characters longer than just a CR/LF
    expect(newName.length).toBeGreaterThan(5);
});
test('Test generateName sunny day for current construct', () => {
    const stack = new cdk.Stack(undefined, "some-new-id");
    const newName = defaults.generateName(stack, "");
    expect(newName.length).toBeGreaterThan(5);
});
test('Test generateName sunny day for child construct', () => {
    const stack = new cdk.Stack(undefined, "some-new-id");
    const newName = defaults.generateName(stack, "child");
    expect(newName.length).toBeGreaterThan(5);
    expect(newName.includes(newName)).toBe(true);
});
test('Test generateName longer than 64 characters', () => {
    const stack = new cdk.Stack(undefined, "some-new-id");
    const seventyCharacterName = '123456789-123456789-123456789-123456789-123456789-123456789-123456789-';
    const newName = defaults.generateName(stack, seventyCharacterName);
    expect(newName.length).toEqual(64);
});
test('Test generateName uniqueness', () => {
    const stackOne = new cdk.Stack(undefined, "some-new-id");
    const stackTwo = new cdk.Stack(undefined, "other-id");
    const nameOne = defaults.generateName(stackOne, "");
    const nameTwo = defaults.generateName(stackTwo, "");
    expect(nameOne === nameTwo).toBe(false);
});
//# sourceMappingURL=data:application/json;base64,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