"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("..");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const __1 = require("..");
// --------------------------------------------------------------
// Test construct with default props
// --------------------------------------------------------------
test('Test construct with default props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    defaults.buildWebacl(stack, 'REGIONAL', {});
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        Scope: "REGIONAL",
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesBotControlRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesBotControlRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesBotControlRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesKnownBadInputsRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesKnownBadInputsRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesKnownBadInputsRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 2,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAnonymousIpList",
                OverrideAction: {
                    None: {}
                },
                Priority: 3,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAnonymousIpList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAnonymousIpList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAmazonIpReputationList",
                OverrideAction: {
                    None: {}
                },
                Priority: 4,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAmazonIpReputationList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAmazonIpReputationList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAdminProtectionRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 5,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAdminProtectionRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAdminProtectionRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesSQLiRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 6,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesSQLiRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesSQLiRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::WAFv2::WebACL', 1);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toCountResources('AWS::WAFv2::WebACLAssociation', 0);
});
// --------------------------------------------------------------
// Test deployment w/ user provided custom properties
// --------------------------------------------------------------
test('Test deployment w/ user provided custom properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    const props = {
        scope: 'CLOUDFRONT',
        defaultAction: {
            allow: {}
        },
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
        rules: [
            defaults.wrapManagedRuleSet("AWSManagedRulesCommonRuleSet", "AWS", 0),
            defaults.wrapManagedRuleSet("AWSManagedRulesWordPressRuleSet", "AWS", 1),
        ]
    };
    defaults.buildWebacl(stack, 'CLOUDFRONT', {
        webaclProps: props
    });
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        Scope: "CLOUDFRONT",
        VisibilityConfig: {
            CloudWatchMetricsEnabled: false,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesWordPressRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesWordPressRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesWordPressRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing WAF web ACL provided
// --------------------------------------------------------------
test('Test deployment w/ existing WAF web ACL provided', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Build WAF web ACL
    const testWaf = __1.buildWebacl(stack, 'CLOUDFRONT', {});
    const newWaf = defaults.buildWebacl(stack, 'CLOUDFRONT', {
        existingWebaclObj: testWaf
    });
    expect(newWaf).toBe(testWaf);
});
//# sourceMappingURL=data:application/json;base64,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