# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['regta_period']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'regta-period',
    'version': '0.1.0',
    'description': 'Moment-independent periods',
    'long_description': '# regta-period\n\n**Library to make moment-independent periods in python.\nIt\'s designed especially for [Regta Framework](https://github.com/SKY-ALIN/regta), \nbut with an ability to use it independently.**\n\n[![versions](https://img.shields.io/pypi/pyversions/regta-period.svg)](https://github.com/SKY-ALIN/regta-period)\n![Tests](https://github.com/SKY-ALIN/regta-period/actions/workflows/tests.yml/badge.svg)\n![Code Quality](https://github.com/SKY-ALIN/regta-period/actions/workflows/code-quality.yml/badge.svg)\n[![codecov](https://codecov.io/gh/SKY-ALIN/regta-period/branch/main/graph/badge.svg?token=NR7AKLXN5H)](https://codecov.io/gh/SKY-ALIN/regta-period)\n[![PyPI version](https://badge.fury.io/py/regta-period.svg)](https://pypi.org/project/regta-period/)\n[![license](https://img.shields.io/github/license/SKY-ALIN/regta-period.svg)](https://github.com/SKY-ALIN/regta-period/blob/main/LICENSE)\n\n## Moment-Independence Idea Explanation\n\nThis term in this context means that relying on this approach we can get the time to time \npoints regardless of the points in which we are.\n```\n        |-----------------|\n        t1       t2     moment\n--------|--------|--------|--------> time\n                 |--------|\n```\n\nWhereas with the standard intervals like `datetime.timedelta`, we get an unnecessary offset:\n```\n        |-----------------|\n        t1       t2     moment\n--------|--------|--------|--------|--------> time\n                 |-----------------|\n```\n\nFor example, it is important in the context of the job scheduler, because when the\nscheduler is redeployed or restarted, you can get an unnecessary time shift or\nunnecessary execution of the job.\n\n## Math Explanation Of Moment-Independence\n\n### Regular Offset\n\nRegular offset is the same as python\'s `timedelta` shift e.g. once per $n_1$ days,\n$n_2$ hours, $n_3$ minutes, $n_4$ seconds, but with the moment-independence idea.\n\nEssentially, it works as a remainder of the time division from the Unix epoch.\nLet $t_{unix}$ be the moment of the Unix epoch, a moment that we can get a grip on.\n$t$ is the current moment. Then, the time since epoch is:\n\n$$\\ \\Delta t = t - t_{unix} $$\n\nLet $T$ be our regular period. Thus, to calculate time until the next moment we must subtract\nfrom our period the remainder of the division by the period. Final function to calculate\ntime until the next moment since current looks following:\n\n$$\\ f(t) = T - ( \\Delta t \\mod T ) = T - ( ( t - t_{unix} ) \\mod T ) $$\n\n### Time Offset\n\nTime offset is stating the exact time e.g. at 9 pm, at 12 am, at 16:30, etc.\nIt works as a shift of the starting point in the exact time and time zone:\n\n$$\\ t_{unix} + \\Delta t_{time} + \\Delta t_{tz} $$\n\nNote that it\'s not possible to combine exact time and short regular intervals such as\nhours, minutes, and seconds.\n\n### Time Windows\n\nTime window is a static time frame in which the result should be included e.g. every Monday, every June, etc. \nA window may be from $t_{min}$ to $t_{max}$, then function result must be included in this interval:\n\n$$ t + f(t) \\in [t_{min}, t_{max}] $$\n\nIf the expression above is true, it means that the result is included in the time window, and the result is correct. \nIf don\'t, we calculate the result from the maximum and calculate the next time window until we find a match:\n\n$$ t + f(t) \\notin [t_{min_n}, t_{max_n}] \\longrightarrow f(t_{max_n}); [t_{min_{n+1}}, t_{max_{n+1}}] $$\n\n## Installation\n\nInstall using `pip install regta-period` or `poetry add regta-period`\n\nIf you use python < 3.9, then also install backports: `pip install "backports.zoneinfo[tzdata]"`\n\n## Examples\n\nThere are two ways to create periods: old school style and hipster style.\n\n```python\nfrom datetime import datetime\nfrom zoneinfo import ZoneInfo\nfrom regta_period import Period\n\n# Hipster style\np = Period().every(3).days.at("17:00").by("Europe/Moscow")\n\n# Old school style\np = Period(days=3, time="17:00", timezone=ZoneInfo("Europe/Moscow"))\n\n# <Period: regular_offset=259200.0s, time_offset=61200s, timezone=Europe/Moscow>\n# Every 3 days at 5 pm by Moscow time\n\nt = datetime.now(tz=ZoneInfo("Europe/Moscow"))\nnext_moment: datetime = p.get_next(t)  # f(t) + t\n```\n\nYou also may combine a few periods to a single object with the same interface:\n\n```python\nfrom datetime import datetime\nfrom regta_period import Period, PeriodAggregation, Weekdays\n\n# Hipster style\np = Period().on.weekdays.at("18:00") | Period().on.weekends.at("21:00")\n# You also may replace `|` with `.OR` to write shorter and more human-readable code\np = Period().on.weekdays.at("18:00").OR.on.weekends.at("21:00")\n\n# Old school style\np = PeriodAggregation(\n    Period(\n        weekdays=[Weekdays.MONDAY, Weekdays.TUESDAY, Weekdays.WEDNESDAY, Weekdays.THURSDAY, Weekdays.FRIDAY],\n        time="18:00",\n    ),\n    Period(\n        weekdays=[Weekdays.SATURDAY, Weekdays.SUNDAY],\n        time="21:00",\n    ),\n)\n\n# All of the above will the same result:\n# <PeriodAggregation: <Period: regular_offset=86400.0s, time_offset=64800s, weekdays=Tuesday,Monday,Thursday,Wednesday,Friday> OR <Period: regular_offset=86400.0s, time_offset=75600s, weekdays=Sunday,Saturday>>\n# At 6 pm on weekdays (Monday-Friday) and at 9 pm on weekends (Saturday-Sunday)\n\nt = datetime.now()\ntimedelta_to_the_next_moment = p.get_interval(t)  # f(t)\n```\n\n---\n\nFull documentation and reference are available at \n[regta-period.alinsky.tech](https://regta-period.alinsky.tech)\n',
    'author': 'Vladimir Alinsky',
    'author_email': 'Vladimir@Alinsky.tech',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://regta-period.alinsky.tech',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7.2,<4.0',
}


setup(**setup_kwargs)
