"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityConfiguration = exports.JobBookmarksEncryptionMode = exports.CloudWatchEncryptionMode = exports.S3EncryptionMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("aws-cdk-lib/aws-kms");
const cdk = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
var S3EncryptionMode;
(function (S3EncryptionMode) {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    S3EncryptionMode["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    S3EncryptionMode["KMS"] = "SSE-KMS";
})(S3EncryptionMode = exports.S3EncryptionMode || (exports.S3EncryptionMode = {}));
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
var CloudWatchEncryptionMode;
(function (CloudWatchEncryptionMode) {
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    CloudWatchEncryptionMode["KMS"] = "SSE-KMS";
})(CloudWatchEncryptionMode = exports.CloudWatchEncryptionMode || (exports.CloudWatchEncryptionMode = {}));
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
var JobBookmarksEncryptionMode;
(function (JobBookmarksEncryptionMode) {
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    JobBookmarksEncryptionMode["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(JobBookmarksEncryptionMode = exports.JobBookmarksEncryptionMode || (exports.JobBookmarksEncryptionMode = {}));
/**
 * A security configuration is a set of security properties that can be used by AWS Glue to encrypt data at rest.
 *
 * The following scenarios show some of the ways that you can use a security configuration.
 * - Attach a security configuration to an AWS Glue crawler to write encrypted Amazon CloudWatch Logs.
 * - Attach a security configuration to an extract, transform, and load (ETL) job to write encrypted Amazon Simple Storage Service (Amazon S3) targets and encrypted CloudWatch Logs.
 * - Attach a security configuration to an ETL job to write its jobs bookmarks as encrypted Amazon S3 data.
 * - Attach a security configuration to a development endpoint to write encrypted Amazon S3 targets.
 */
class SecurityConfiguration extends cdk.Resource {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, {
            physicalName: props.securityConfigurationName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_SecurityConfigurationProps(props);
        if (!props.s3Encryption && !props.cloudWatchEncryption && !props.jobBookmarksEncryption) {
            throw new Error('One of cloudWatchEncryption, jobBookmarksEncryption or s3Encryption must be defined');
        }
        const kmsKeyCreationRequired = (props.s3Encryption && props.s3Encryption.mode === S3EncryptionMode.KMS && !props.s3Encryption.kmsKey) ||
            (props.cloudWatchEncryption && !props.cloudWatchEncryption.kmsKey) ||
            (props.jobBookmarksEncryption && !props.jobBookmarksEncryption.kmsKey);
        const autoCreatedKmsKey = kmsKeyCreationRequired ? new kms.Key(this, 'Key') : undefined;
        let cloudWatchEncryption;
        if (props.cloudWatchEncryption) {
            this.cloudWatchEncryptionKey = props.cloudWatchEncryption.kmsKey || autoCreatedKmsKey;
            cloudWatchEncryption = {
                cloudWatchEncryptionMode: props.cloudWatchEncryption.mode,
                kmsKeyArn: (_b = this.cloudWatchEncryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn,
            };
        }
        let jobBookmarksEncryption;
        if (props.jobBookmarksEncryption) {
            this.jobBookmarksEncryptionKey = props.jobBookmarksEncryption.kmsKey || autoCreatedKmsKey;
            jobBookmarksEncryption = {
                jobBookmarksEncryptionMode: props.jobBookmarksEncryption.mode,
                kmsKeyArn: (_c = this.jobBookmarksEncryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
            };
        }
        let s3Encryptions;
        if (props.s3Encryption) {
            if (props.s3Encryption.mode === S3EncryptionMode.KMS) {
                this.s3EncryptionKey = props.s3Encryption.kmsKey || autoCreatedKmsKey;
            }
            // NOTE: CloudFormations errors out if array is of length > 1. That's why the props don't expose an array
            s3Encryptions = [{
                    s3EncryptionMode: props.s3Encryption.mode,
                    kmsKeyArn: (_d = this.s3EncryptionKey) === null || _d === void 0 ? void 0 : _d.keyArn,
                }];
        }
        const resource = new aws_glue_1.CfnSecurityConfiguration(this, 'Resource', {
            name: props.securityConfigurationName,
            encryptionConfiguration: {
                cloudWatchEncryption,
                jobBookmarksEncryption,
                s3Encryptions,
            },
        });
        this.securityConfigurationName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Creates a Connection construct that represents an external security configuration.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param securityConfigurationName name of external security configuration.
     */
    static fromSecurityConfigurationName(scope, id, securityConfigurationName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.securityConfigurationName = securityConfigurationName;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecurityConfiguration = SecurityConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SecurityConfiguration[_a] = { fqn: "@aws-cdk/aws-glue-alpha.SecurityConfiguration", version: "2.19.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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