"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'fixed retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.ONE_WEEK
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 7
        }));
        test.done();
    },
    'default retention'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: 731
        }));
        test.done();
    },
    'infinite retention/dont delete log group by default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: lib_1.RetentionDays.INFINITE,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'infinite retention via legacy method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            // Don't know why TypeScript doesn't complain about passing Infinity to
            // something where an enum is expected, but better keep this behavior for
            // existing clients.
            retention: Infinity
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Retain",
                    UpdateReplacePolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'will delete log group if asked to'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.LogGroup(stack, 'LogGroup', {
            retention: Infinity,
            removalPolicy: core_1.RemovalPolicy.DESTROY
        });
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                LogGroupF5B46931: {
                    Type: "AWS::Logs::LogGroup",
                    DeletionPolicy: "Delete",
                    UpdateReplacePolicy: "Delete"
                }
            }
        }));
        test.done();
    },
    'export/import'(test) {
        // GIVEN
        const stack2 = new core_1.Stack();
        // WHEN
        const imported = lib_1.LogGroup.fromLogGroupArn(stack2, 'lg', 'arn:aws:logs:us-east-1:123456789012:log-group:my-log-group');
        imported.addStream('MakeMeAStream');
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::Logs::LogStream', {
            LogGroupName: "my-log-group"
        }));
        test.done();
    },
    'extractMetric'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            LogGroupName: { Ref: "LogGroupF5B46931" },
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'extractMetric allows passing in namespaces with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        // WHEN
        const metric = lg.extractMetric('$.myField', 'MyNamespace/MyService', 'Field');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::MetricFilter', {
            FilterPattern: "{ $.myField = \"*\" }",
            MetricTransformations: [
                {
                    MetricName: "Field",
                    MetricNamespace: "MyNamespace/MyService",
                    MetricValue: "$.myField"
                }
            ]
        }));
        test.equal(metric.namespace, 'MyNamespace/MyService');
        test.equal(metric.metricName, 'Field');
        test.done();
    },
    'grant'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const lg = new lib_1.LogGroup(stack, 'LogGroup');
        const user = new iam.User(stack, 'User');
        // WHEN
        lg.grantWrite(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ["logs:CreateLogStream", "logs:PutLogEvents"],
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["LogGroupF5B46931", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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