"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue
            }
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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