"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerService = exports.Database = exports.KeyCloak = void 0;
const certmgr = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class KeyCloak extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : getOrCreateVpc(this);
        this.db = this.addDatabase();
        this.addKeyCloakContainerService({
            database: this.db,
            vpc: this.vpc,
            publicSubnets: props.publicSubnets,
            privateSubnets: props.privateSubnets,
            keycloakSecret: this._generateKeycloakSecret(),
            certificate: certmgr.Certificate.fromCertificateArn(this, 'ACMCert', props.certificateArn),
            bastion: props.bastion,
            nodeCount: props.nodeCount,
        });
    }
    /**
     * @stability stable
     */
    addDatabase() {
        return new Database(this, 'Database', {
            vpc: this.vpc,
        });
    }
    /**
     * @stability stable
     */
    addKeyCloakContainerService(props) {
        return new ContainerService(this, 'KeyCloakContainerSerivce', props);
    }
    _generateKeycloakSecret() {
        return new secretsmanager.Secret(this, 'KCSecret', {
            generateSecretString: {
                generateStringKey: 'password',
                excludePunctuation: true,
                passwordLength: 12,
                secretStringTemplate: JSON.stringify({ username: 'keycloak' }),
            },
        });
    }
}
exports.KeyCloak = KeyCloak;
/**
 * @stability stable
 */
class Database extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this._mysqlListenerPort = 3306;
        const dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
            vpc: props.vpc,
            vpcSubnets: props.databaseSubnets,
            engine: rds.DatabaseInstanceEngine.mysql({
                version: rds.MysqlEngineVersion.VER_8_0_21,
            }),
            credentials: rds.Credentials.fromGeneratedSecret('admin'),
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : new ec2.InstanceType('r5.large'),
            parameterGroup: rds.ParameterGroup.fromParameterGroupName(this, 'ParameterGroup', 'default.mysql8.0'),
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.secret = dbInstance.secret;
        // allow internally from the same security group
        dbInstance.connections.allowInternally(ec2.Port.tcp(this._mysqlListenerPort));
        // allow from the whole vpc cidr
        dbInstance.connections.allowFrom(ec2.Peer.ipv4(props.vpc.vpcCidrBlock), ec2.Port.tcp(this._mysqlListenerPort));
        this.dbinstance = dbInstance;
        this.vpc = props.vpc;
        this.clusterEndpointHostname = dbInstance.dbInstanceEndpointAddress;
        this.clusterIdentifier = dbInstance.instanceIdentifier;
        printOutput(this, 'clusterEndpointHostname', this.clusterEndpointHostname);
        printOutput(this, 'clusterIdentifier', this.clusterIdentifier);
        if (this.dbinstance.secret) {
            printOutput(this, 'DBSecretArn', this.dbinstance.secret.secretArn);
        }
    }
}
exports.Database = Database;
/**
 * @stability stable
 */
class ContainerService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const vpc = props.vpc;
        const cluster = new ecs.Cluster(this, 'Cluster', { vpc });
        const taskRole = new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com')),
        });
        const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDef', {
            cpu: 4096,
            memoryLimitMiB: 30720,
            executionRole: taskRole,
        });
        const logGroup = new logs.LogGroup(this, 'LogGroup', {
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        // bootstrap container that creates the database if not exist
        const bootstrap = taskDefinition.addContainer('bootstrap', {
            essential: false,
            image: ecs.ContainerImage.fromRegistry('public.ecr.aws/ubuntu/mysql:latest'),
            environment: {
                DB_NAME: 'keycloak',
                DB_USER: 'admin',
                DB_ADDR: props.database.clusterEndpointHostname,
            },
            secrets: {
                DB_PASSWORD: ecs.Secret.fromSecretsManager(props.database.secret, 'password'),
            },
            command: [
                'sh', '-c',
                'mysql -u$DB_USER -p$DB_PASSWORD -h$DB_ADDR -e "CREATE DATABASE IF NOT EXISTS $DB_NAME"',
            ],
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'bootstrap',
                logGroup,
            }),
        });
        const kc = taskDefinition.addContainer('keycloak', {
            image: ecs.ContainerImage.fromRegistry('jboss/keycloak:12.0.2'),
            environment: {
                DB_ADDR: props.database.clusterEndpointHostname,
                DB_DATABASE: 'keycloak',
                DB_PORT: '3306',
                DB_USER: 'admin',
                DB_VENDOR: 'mysql',
                JDBC_PARAMS: 'useSSL=false',
            },
            secrets: {
                DB_PASSWORD: ecs.Secret.fromSecretsManager(props.database.secret, 'password'),
                KEYCLOAK_PASSWORD: ecs.Secret.fromSecretsManager(props.keycloakSecret, 'password'),
            },
            logging: ecs.LogDrivers.awsLogs({
                streamPrefix: 'keycloak',
                logGroup,
            }),
        });
        kc.addPortMappings({
            containerPort: 8080,
        });
        kc.addContainerDependencies({
            container: bootstrap,
            condition: ecs.ContainerDependencyCondition.SUCCESS,
        });
        this.service = new ecs.FargateService(this, 'Service', {
            cluster,
            taskDefinition,
            circuitBreaker: props.circuitBreaker ? { rollback: true } : undefined,
            desiredCount: (_a = props.nodeCount) !== null && _a !== void 0 ? _a : 1,
            healthCheckGracePeriod: cdk.Duration.seconds(120),
        });
        const alb = new elbv2.ApplicationLoadBalancer(this, 'ALB', {
            vpc,
            vpcSubnets: props.publicSubnets,
            internetFacing: true,
        });
        printOutput(this, 'EndpointURL', alb.loadBalancerDnsName);
        const listener = alb.addListener('HttpListener', {
            protocol: elbv2.ApplicationProtocol.HTTPS,
            certificates: [{ certificateArn: props.certificate.certificateArn }],
        });
        listener.addTargets('ECSTarget', {
            targets: [this.service],
            // set slow_start.duration_seconds to 60
            // see https://docs.aws.amazon.com/cli/latest/reference/elbv2/modify-target-group-attributes.html
            slowStart: cdk.Duration.seconds(60),
            stickinessCookieDuration: cdk.Duration.days(1),
            port: 8080,
            protocol: elbv2.ApplicationProtocol.HTTP,
        });
        // allow task execution role to read the secrets
        props.database.secret.grantRead(taskDefinition.executionRole);
        props.keycloakSecret.grantRead(taskDefinition.executionRole);
        // allow ecs task connect to database
        props.database.dbinstance.connections.allowDefaultPortFrom(this.service);
        // create a bastion host
        if (props.bastion === true) {
            const bast = new ec2.BastionHostLinux(this, 'Bast', {
                vpc,
                instanceType: new ec2.InstanceType('m5.large'),
            });
            props.database.dbinstance.connections.allowDefaultPortFrom(bast);
        }
    }
}
exports.ContainerService = ContainerService;
/**
 * Create or import VPC
 * @param scope the cdk scope
 */
function getOrCreateVpc(scope) {
    // use an existing vpc or create a new one
    return scope.node.tryGetContext('use_default_vpc') === '1' ?
        ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
        scope.node.tryGetContext('use_vpc_id') ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
            new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
}
function printOutput(scope, id, key) {
    new cdk.CfnOutput(scope, id, { value: String(key) });
}
//# sourceMappingURL=data:application/json;base64,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