import asyncio

import click
from fastmcp import FastMCP

from artl_mcp.client import run_client
from artl_mcp.tools import (
    clean_text,
    doi_to_pmid,
    # PubMed utilities tools
    extract_doi_from_url,
    extract_pdf_text,
    get_abstract_from_pubmed_id,
    # DOIFetcher-based tools
    get_doi_fetcher_metadata,
    # Original tools
    get_doi_metadata,
    get_doi_text,
    get_full_text_from_bioc,
    get_full_text_from_doi,
    get_full_text_info,
    get_pmcid_text,
    get_pmid_from_pmcid,
    get_pmid_text,
    get_text_from_pdf_url,
    get_unpaywall_info,
    pmid_to_doi,
    # Search tools
    search_papers_by_keyword,
    search_recent_papers,
)


def create_mcp():
    """Create the FastMCP server instance and register tools."""
    mcp = FastMCP("all-roads-to-literature")

    # Register all tools
    # Original tools
    mcp.tool(get_doi_metadata)
    mcp.tool(get_abstract_from_pubmed_id)

    # DOIFetcher-based tools (require email)
    mcp.tool(get_doi_fetcher_metadata)
    mcp.tool(get_unpaywall_info)
    mcp.tool(get_full_text_from_doi)
    mcp.tool(get_full_text_info)
    mcp.tool(get_text_from_pdf_url)
    mcp.tool(clean_text)

    # Standalone tools
    mcp.tool(extract_pdf_text)

    # PubMed utilities tools
    mcp.tool(extract_doi_from_url)
    mcp.tool(doi_to_pmid)
    mcp.tool(pmid_to_doi)
    mcp.tool(get_doi_text)
    mcp.tool(get_pmid_from_pmcid)
    mcp.tool(get_pmcid_text)
    mcp.tool(get_pmid_text)
    mcp.tool(get_full_text_from_bioc)

    # Search tools
    mcp.tool(search_papers_by_keyword)
    mcp.tool(search_recent_papers)

    return mcp


# Server instance
mcp = create_mcp()


@click.command()
@click.option("--server", is_flag=True, help="Start the MCP server.")
@click.option("--doi-query", type=str, help="Run a direct query (DOI string).")
def cli(server, doi_query):
    """Run All Roads to Literature MCP tool or server."""
    if server:
        # Run the server over stdio
        mcp.run()
    elif doi_query:
        # Run the client in asyncio
        asyncio.run(run_client(doi_query, mcp))
    else:
        click.echo(cli.get_help(click.Context(cli)))


if __name__ == "__main__":
    cli()
