# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['graal',
 'graal.backends',
 'graal.backends.core',
 'graal.backends.core.analyzers',
 'graal.bin',
 'tests',
 'tests.data']

package_data = \
{'': ['*']}

install_requires = \
['bandit>=1.4.0',
 'cloc>=0.2.5,<0.3.0',
 'execnet>=1.9.0,<2.0.0',
 'flake8>=3.7.7',
 'grimoirelab-toolkit>=0.3',
 'lizard==1.16.6',
 'networkx>=2.1',
 'perceval>=0.19',
 'pydot>=1.2.4',
 'pylint>=1.8.4']

entry_points = \
{'console_scripts': ['graal = graal.bin.graal:main']}

setup_kwargs = {
    'name': 'graal',
    'version': '0.3.2rc9',
    'description': 'A generic source code analyzer',
    'long_description': '# Graal: a Generic Repository AnALyzer [![Build Status](https://github.com/chaoss/grimoirelab-graal/workflows/tests/badge.svg)](https://github.com/chaoss/grimoirelab-graal/actions?query=workflow:tests+branch:master+event:push) [![Coverage Status](https://coveralls.io/repos/github/chaoss/grimoirelab-graal/badge.svg?branch=master)](https://coveralls.io/github/chaoss/grimoirelab-graal?branch=master)\n\nGraal leverages on the Git backend of [Perceval](https://github.com/chaoss/grimoirelab-perceval) and enhances it to set up ad-hoc\nsource code analysis. Thus, it fetches the commits from a Git repository and provides a mechanism to plug third party tools/libraries focused on source code analysis.\n\n## How it works\nThe Perceval Git backend creates a local mirror of a Git repository (local or remote), fetches the metadata of commits in chronological order and returns them as a list of JSON documents\n(one per commit). Graal leverages on the incremental functionalities provided by the Git backend and enhances the logic to handle Git repositories by creating a\nworking tree to perform checkout operations (which are not possible on a Git mirror). Graal intercepts each JSON document and enables the user to perform the following steps:\n- **Filter.** The filtering is used to select or discard commits based on the information available in the JSON document and/or via the Graal parameters. For any selected commit,\nGraal executes a checkout on the working tree using the commit hash, thus setting the state of the working tree at that given revision.\n- **Analyze.** The analysis takes the JSON document and the current working tree and enables the user to set up ad-hoc source code analysis by plugging existing tools through system calls\nor their Python interfaces, when possible. The results of the analysis are parsed and manipulated by the user and then automatically embedded in the JSON document. It is worth\nnoting that in this step the user can rely on some predefined functionalities of Graal to deal with the repository snapshot (e.g., listing files, creating archives).\n- **Post-process.** In the final step, the inflated JSON document can be optionally processed to alter (e.g., renaming, removing) its attributes, thus granting the user complete control over the output of Graal executions.\n\nSeveral parameters (inherited from the [Git backend](http://perceval.readthedocs.io/en/latest/perceval.backends.core.html#module-perceval.backends.core.git)) are available to control the execution; for instance, **from_date** and **to_date**\nallow to select commits authored since and before a given date, **branches** allows to fetch commits only from specific branches,\nand **latest_items** returns only those commits which are new since the last fetch operation. Graal includes additional parameters to drive\nthe analysis to filter in/out files and directories in the repository (**in_paths** and **out_paths**), set the **entrypoint**\nand define the **details** level of the analysis (useful when analyzing large software projects).\n\n## Requirements\n- [lizard](https://github.com/terryyin/lizard)==1.16.6\n- [perceval](https://github.com/chaoss/grimoirelab-perceval)>=0.9.6\n- [pylint](https://github.com/PyCQA/pylint)>=1.8.4\n- [flake8](https://github.com/PyCQA/flake8)>=3.7.7\n- [networkx](https://github.com/networkx/networkx)>=2.1\n- [pydot](https://github.com/pydot/pydot)>=1.2.4\n- [bandit](https://github.com/PyCQA/bandit)>=1.4.0\n- [grimoirelab-toolkit](https://github.com/chaoss/grimoirelab-toolkit)>=0.1.4\n- [cloc](http://cloc.sourceforge.net/)\n- [scc](https://github.com/boyter/scc)\n- [nomos](https://github.com/fossology/fossology/tree/master/src/nomos) \n- [scancode](https://github.com/nexB/scancode-toolkit) \n- [github-linguist](https://github.com/github/linguist) <=7.15\n\n### How to install/create the executables:\n- **Cloc**\n  ```\n  $> sudo apt-get install cloc\n  ```\n\n- **SCC**\n\nA tool similar to cloc - for counting physical the lines of code, blank lines, comment lines, and physical lines of source code in many programming languages and COCOMO estimates written in pure Go.\n\n  ```\n  $> go get -u github.com/boyter/scc/\n  ```\n\n\n- **Nomos**\n\nMaybe you\'ll need to install some packages for compiling the tool.\nFor example, in Debian, likely you\'ll need:\n\n```\nsudo apt-get install pkg-config libglib2.0-dev libjson-c-dev libpq-dev\n```\n\n- For compiling the tool (`nomossa`):\n\n```\n$> git clone https://github.com/fossology/fossology\n$> cd <...>/fossology/src/nomos/agent\n$> make -f Makefile.sa FO_LDFLAGS="-lglib-2.0 -lpq  -lglib-2.0 -ljson-c -lpthread -lrt"\n```\n\n- **ScanCode**\n\n```\ngit clone https://github.com/nexB/scancode-toolkit.git\ncd scancode-toolkit\ngit checkout -b test_scancli 96069fd84066c97549d54f66bd2fe8c7813c6b52\n./scancode --help\n```\n\n   **Note**: We\'re now using a clone of scancode-toolkit instead of a release, as the latest release is of 15th February 2019 and the `scancli.py` script (required for execution of scancode_cli) was incroporated later i.e 5th March 2019 and there hasn\'t been a release since.\n\n- **ScanCode Cli**\n\nAfter successfully executing the above mentioned steps, (if required) we have to install python modules: `simplejson` and `execnet`, for the execution of `scancode_cli` analyzer.\n\n```\npip install simplejson execnet\n```\n\n\n##  How to install/uninstall\nGraal is being developed and tested mainly on GNU/Linux platforms. Thus it is very likely it will work out of the box\non any Linux-like (or Unix-like) platform, upon providing the right version of Python (3.5, 3.6).\n\n\n**To install**, run:\n```\n$> git clone https://github.com/chaoss/grimoirelab-graal.git\n$> python3 setup.py build\n$> python3 setup.py install\n```\n\n**To uninstall**, run:\n```\n$> pip3 uninstall graal\n```\n\n## Backends\nSeveral backends have been developed to assess the genericity of Graal. Those backends leverage on source code analysis\ntools, where executions are triggered via system calls or their Python interfaces. In the current status, the backends\nmostly target Python code, however other backends can be easily developed to cover other programming languages. The\ncurrently available backends are:\n- **CoCom** gathers data about code complexity (e.g., cyclomatic complexity, LOC) from projects written in popular programming languages such as: C/C++, Java, Scala, JavaScript, Ruby, Python, Lua and Golang. It leverages on [Cloc](http://cloc.sourceforge.net/), [Lizard](https://github.com/terryyin/lizard) and [scc](https://github.com/boyter/scc). The tool can be exectued at `file` and `repository` levels activated with the help of category: `code_complexity_lizard_file` or `code_complexity_lizard_repository`.\n- **CoDep** extracts package and class dependencies of a Python module and serialized them as JSON structures, composed of edges and nodes, thus easing the bridging with front-end technologies for graph visualizations. It combines [PyReverse](https://pypi.org/project/pyreverse/) and [NetworkX](https://networkx.github.io/).\n- **CoQua** retrieves code quality insights, such as checks about line-code’s length, well-formed variable names, unused imported modules and code clones. It uses [PyLint](https://www.pylint.org/) and [Flake8](http://flake8.pycqa.org/en/latest/index.html). The tools can be activated by passing the corresponding category: `code_quality_pylint` or `code_quality_flake8`.\n- **CoVuln** scans the code to identify security vulnerabilities such as potential SQL and Shell injections, hard-coded passwords and weak cryptographic key size. It relies on [Bandit](https://github.com/PyCQA/bandit).\n- **CoLic** scans the code to extract license & copyright information. It currently supports [Nomos](https://github.com/fossology/fossology/tree/master/src/nomos) and [ScanCode](https://github.com/nexB/scancode-toolkit). They can be activated by passing the corresponding category: `code_license_nomos`, `code_license_scancode`, or `code_license_scancode_cli`.\n- **CoLang** gathers insights about code language distribution of a git repository. It relies on [Linguist](https://github.com/github/linguist) and [Cloc](http://cloc.sourceforge.net/) tools. They can be activated by passing the corresponding category: `code_language_linguist` or `code_language_cloc`.\n\n### How to develop a backend\nCreating your own backend is pretty easy, you only need to redefine the following methods of Graal:\n- **_filter_commit.** This method is used to select or discard commits based on the information available in the JSON document\nand/or via the Graal parameters (e.g., the commits authored by a given user or targeting a given software component).\nFor any selected commit, Graal executes a checkout on the working tree using the commit hash, thus setting the state of\nthe working tree at that given revision.\n- **_analyze.** This method takes the document and the current working tree and allows to connect existing tools through system calls or their Python interfaces, when possible.\nThe results of the analysis, parsed and manipulated by the user, are automatically embedded in the JSON document.\n- **_post.** This method allows to alter (e.g., renaming, removing) the attributes of the inflated JSON documents.\n\n## How to use\n\nGraal can be used from command line or directly from Python, both usages are described below.\n\n### From command line\nLaunching Graal from command line does not require much effort, but only some basic knowledge of GNU/Linux shell commands.\n\nThe example below shows how easy it is to fetch code complexity information from a Git repository. The **CoCom** backend\nrequires the URL where the repository is located (_https://github.com/chaoss/grimoirelab-perceval_) and the local path\nwhere to mirror the repository (_/tmp/graal-cocom_). Then, the JSON documents produced are redirected to the file _graal-cocom.test_.\n\n- **CoCom Backend**\n\n```\n$ graal cocom https://github.com/chaoss/grimoirelab-perceval --git-path /tmp/graal-cocom > /graal-cocom.test\nStarting the quest for the Graal.\nGit worktree /tmp/... created!\nFetching commits: ...\nGit worktree /tmp/... deleted!\nFetch process completed: .. commits inspected\nQuest completed.\n```\n\n- **CoLic Backend**\n\n```\ngraal colic https://github.com/chaoss/grimoirelab-toolkit --git-path /tmp/scancode_cli --exec-path /home/scancode-toolkit/etc/scripts/scancli.py --category code_license_scancode_cli\nStarting the quest for the Graal.\nGit worktree /tmp/... created!\nFetching commits: ...\nGit worktree /tmp/... deleted!\nFetch process completed: .. commits inspected\nQuest completed.\n```\n\nIn the above example, we\'re using scancode_cli analyzer. Similarly, we can use the scancode analyzer by providing the category as `code_license_scancode` and it\'s corresponding executable path.\n\n### From Python\nGraal’s functionalities can be embedded in Python scripts. Again, the effort of using Graal is minimum. In this case the user\nonly needs some knowledge of Python scripting. The example below shows how to use Graal in a script.\n\nThe **graal.backends.core.cocom** module is imported at the beginning of the file, then the **repo_uri** and **repo_dir** variables\nare set to the URI of the Git repository and the local path where to mirror it. These variables are used to initialize a\n**CoCom class object**. In the last line of the script, the commits inflated with the result of the analysis are retrieved\nusing the fetch method. The fetch method inherits its argument from **Perceval**, thus it optionally accept two Datetime\nobjects to gather only those commits after and before a given date, a list of branches to focus on specific development\nactivities, and a flag to collect the commits available after the last execution.\n\n```\n#! /usr/bin/env python3\nfrom graal.backends.core.cocom import CoCom\n\n# URL for the git repo to analyze\nrepo_uri = ’http://github.com/chaoss/grimoirelab-perceval’\n\n# directory where to mirror the repo\nrepo_dir = ’/tmp/graal-cocom’\n\n# Cocom object initialization\ncc = CoCom(uri=repo_uri, git_path=repo_dir)\n\n# fetch all commits\ncommits = [commit for commit in cc.fetch()]\n```\n\n## How to integrate it with Arthur\n[Arthur](https://github.com/chaoss/grimoirelab-kingarthur) is another tool of the [Grimoirelab ecosystem](https://chaoss.github.io/grimoirelab/). It was originally designed to allow to schedule\nand run Perceval executions at scale through distributed **Redis** queues, and store the obtained results in an **ElasticSearch** database.\n\nArthur has been extended to allow handling Graal tasks, which inherit from Perceval Git tasks. The code to make this extension possible is\navailable at: https://github.com/chaoss/grimoirelab-kingarthur/pull/33.\n\nInformation about Arthur is available at https://github.com/chaoss/grimoirelab-kingarthur.\n',
    'author': 'GrimoireLab Developers',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://chaoss.github.io/grimoirelab/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
