"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const lib_1 = require("../lib");
/**
 * Given a test function which sets the stage and verifies a dependency scenario
 * between two CloudFormation resources, returns two tests which exercise both
 * "construct dependency" (i.e. node.addDependency) and "resource dependency"
 * (i.e. resource.addDependsOn).
 *
 * @param testFunction The test function
 */
function matrixForResourceDependencyTest(testFunction) {
    return {
        'construct dependency'(test) {
            testFunction(test, (source, target) => source.node.addDependency(target));
        },
        'resource dependency'(test) {
            testFunction(test, (source, target) => source.addDependsOn(target));
        },
    };
}
function assertAssemblyDependency(test, assembly, stack, expectedDeps) {
    const stack1Art = assembly.getStackArtifact(stack.artifactId);
    const stack1Deps = stack1Art.dependencies.map(x => x.id);
    test.deepEqual(stack1Deps, expectedDeps);
}
function assertNoDependsOn(test, assembly, stack) {
    let templateText;
    if (!(stack instanceof lib_1.NestedStack)) {
        templateText = JSON.stringify(assembly.getStackArtifact(stack.artifactId).template);
    }
    else {
        templateText = fs.readFileSync(path.join(assembly.directory, stack.templateFile), 'utf-8');
    }
    // verify templates do not have any "DependsOn"
    test.ok(!templateText.includes('DependsOn'));
}
module.exports = {
    'resource dependencies': {
        'between two resources in a top-level stack'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            const r1 = new core_1.CfnResource(stack, 'r1', { type: 'r1' });
            const r2 = new core_1.CfnResource(stack, 'r2', { type: 'r2' });
            // WHEN
            r1.addDependsOn(r2);
            // THEN
            test.deepEqual(app.synth().getStackArtifact(stack.artifactId).template, {
                Resources: { r1: { Type: 'r1', DependsOn: ['r2'] }, r2: { Type: 'r2' } },
            });
            test.done();
        },
        'resource in nested stack depends on a resource in the parent stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const parent = new core_1.Stack(undefined, 'root');
            const nested = new lib_1.NestedStack(parent, 'Nested');
            const resourceInParent = new core_1.CfnResource(parent, 'ResourceInParent', { type: 'PARENT' });
            const resourceInNested = new core_1.CfnResource(nested, 'ResourceInNested', { type: 'NESTED' });
            // WHEN
            addDep(resourceInNested, resourceInParent);
            // THEN: the dependency needs to transfer from the resource within the
            // nested stack to the nested stack resource itself so the nested stack
            // will only be deployed the dependent resource
            assert_1.expect(parent).to(assert_1.haveResource('AWS::CloudFormation::Stack', { DependsOn: ['ResourceInParent'] }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(nested).toMatch({ Resources: { ResourceInNested: { Type: 'NESTED' } } }); // no DependsOn for the actual resource
            test.done();
        }),
        'resource in nested stack depends on a resource in a grandparent stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const grantparent = new core_1.Stack(undefined, 'Grandparent');
            const parent = new lib_1.NestedStack(grantparent, 'Parent');
            const nested = new lib_1.NestedStack(parent, 'Nested');
            const resourceInGrandparent = new core_1.CfnResource(grantparent, 'ResourceInGrandparent', { type: 'GRANDPARENT' });
            const resourceInNested = new core_1.CfnResource(nested, 'ResourceInNested', { type: 'NESTED' });
            // WHEN
            addDep(resourceInNested, resourceInGrandparent);
            // THEN: the dependency needs to transfer from the resource within the
            // nested stack to the *parent* nested stack
            assert_1.expect(grantparent).to(assert_1.haveResource('AWS::CloudFormation::Stack', { DependsOn: ['ResourceInGrandparent'] }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(nested).toMatch({ Resources: { ResourceInNested: { Type: 'NESTED' } } }); // no DependsOn for the actual resource
            test.done();
        }),
        'resource in parent stack depends on resource in nested stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const parent = new core_1.Stack(undefined, 'root');
            const nested = new lib_1.NestedStack(parent, 'Nested');
            const resourceInParent = new core_1.CfnResource(parent, 'ResourceInParent', { type: 'PARENT' });
            const resourceInNested = new core_1.CfnResource(nested, 'ResourceInNested', { type: 'NESTED' });
            // WHEN
            addDep(resourceInParent, resourceInNested);
            // THEN: resource in parent needs to depend on the nested stack
            assert_1.expect(parent).to(assert_1.haveResource('PARENT', {
                DependsOn: [parent.resolve(nested.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }),
        'resource in grantparent stack depends on resource in nested stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const grandparent = new core_1.Stack(undefined, 'Grandparent');
            const parent = new lib_1.NestedStack(grandparent, 'Parent');
            const nested = new lib_1.NestedStack(parent, 'Nested');
            const resourceInGrandparent = new core_1.CfnResource(grandparent, 'ResourceInGrandparent', { type: 'GRANDPARENT' });
            const resourceInNested = new core_1.CfnResource(nested, 'ResourceInNested', { type: 'NESTED' });
            // WHEN
            addDep(resourceInGrandparent, resourceInNested);
            // THEN: resource in grantparent needs to depend on the top-level nested stack
            assert_1.expect(grandparent).to(assert_1.haveResource('GRANDPARENT', {
                DependsOn: [grandparent.resolve(parent.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }),
        'resource in sibling stack depends on a resource in nested stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const app = new core_1.App();
            const stack1 = new core_1.Stack(app, 'Stack1');
            const nested1 = new lib_1.NestedStack(stack1, 'Nested1');
            const resourceInNested1 = new core_1.CfnResource(nested1, 'ResourceInNested', { type: 'NESTED' });
            const stack2 = new core_1.Stack(app, 'Stack2');
            const resourceInStack2 = new core_1.CfnResource(stack2, 'ResourceInSibling', { type: 'SIBLING' });
            // WHEN
            addDep(resourceInStack2, resourceInNested1);
            // THEN: stack2 should depend on stack1 and no "DependsOn" inside templates
            const assembly = app.synth();
            assertAssemblyDependency(test, assembly, stack1, []);
            assertAssemblyDependency(test, assembly, stack2, ['Stack1']);
            assertNoDependsOn(test, assembly, stack1);
            assertNoDependsOn(test, assembly, stack2);
            assertNoDependsOn(test, assembly, nested1);
            test.done();
        }),
        'resource in nested stack depends on a resource in sibling stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const app = new core_1.App();
            const stack1 = new core_1.Stack(app, 'Stack1');
            const nested1 = new lib_1.NestedStack(stack1, 'Nested1');
            const resourceInNested1 = new core_1.CfnResource(nested1, 'ResourceInNested', { type: 'NESTED' });
            const stack2 = new core_1.Stack(app, 'Stack2');
            const resourceInStack2 = new core_1.CfnResource(stack2, 'ResourceInSibling', { type: 'SIBLING' });
            // WHEN
            addDep(resourceInNested1, resourceInStack2);
            // THEN: stack1 should depend on stack2 and no "DependsOn" inside templates
            const assembly = app.synth();
            assertAssemblyDependency(test, assembly, stack1, ['Stack2']);
            assertAssemblyDependency(test, assembly, stack2, []);
            assertNoDependsOn(test, assembly, stack1);
            assertNoDependsOn(test, assembly, stack2);
            assertNoDependsOn(test, assembly, nested1);
            test.done();
        }),
        'resource in nested stack depends on a resource in nested sibling stack': matrixForResourceDependencyTest((test, addDep) => {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack1');
            const nested1 = new lib_1.NestedStack(stack, 'Nested1');
            const nested2 = new lib_1.NestedStack(stack, 'Nested2');
            const resourceInNested1 = new core_1.CfnResource(nested1, 'ResourceInNested1', { type: 'NESTED1' });
            const resourceInNested2 = new core_1.CfnResource(nested2, 'ResourceInNested2', { type: 'NESTED2' });
            // WHEN
            addDep(resourceInNested1, resourceInNested2);
            // THEN: dependency transfered to nested stack resources
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
                DependsOn: [stack.resolve(nested2.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::CloudFormation::Stack', {
                DependsOn: [stack.resolve(nested1.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }),
    },
    'stack dependencies': {
        'top level stack depends on itself'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            // WHEN
            stack.addDependency(stack);
            // THEN
            const assembly = app.synth();
            assertAssemblyDependency(test, assembly, stack, []);
            assertNoDependsOn(test, assembly, stack);
            test.done();
        },
        'nested stack depends on itself'(test) {
            // GIVEN
            const app = new core_1.App();
            const parent = new core_1.Stack(app, 'Parent');
            const nested = new lib_1.NestedStack(parent, 'Nested');
            // WHEN
            nested.addDependency(nested);
            // THEN
            assertNoDependsOn(test, app.synth(), parent);
            test.done();
        },
        'nested stack cannot depend on any of its parents'(test) {
            // GIVEN
            const root = new core_1.Stack();
            const nested1 = new lib_1.NestedStack(root, 'Nested1');
            const nested2 = new lib_1.NestedStack(nested1, 'Nested2');
            // THEN
            test.throws(() => nested1.addDependency(root), /Nested stack 'Default\/Nested1' cannot depend on a parent stack 'Default'/);
            test.throws(() => nested2.addDependency(nested1), /Nested stack 'Default\/Nested1\/Nested2' cannot depend on a parent stack 'Default\/Nested1'/);
            test.throws(() => nested2.addDependency(root), /Nested stack 'Default\/Nested1\/Nested2' cannot depend on a parent stack 'Default'/);
            test.done();
        },
        'any parent stack is by definition dependent on the nested stack so dependency is ignored'(test) {
            // GIVEN
            const root = new core_1.Stack();
            const nested1 = new lib_1.NestedStack(root, 'Nested1');
            const nested2 = new lib_1.NestedStack(nested1, 'Nested2');
            // WHEN
            root.addDependency(nested1);
            root.addDependency(nested2);
            nested1.addDependency(nested2);
            // THEN
            test.done();
        },
        'sibling nested stacks transfer to resources'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const nested1 = new lib_1.NestedStack(stack, 'Nested1');
            const nested2 = new lib_1.NestedStack(stack, 'Nested2');
            // WHEN
            nested1.addDependency(nested2);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
                DependsOn: [stack.resolve(nested2.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'nested stack depends on a deeply nested stack'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const nested1 = new lib_1.NestedStack(stack, 'Nested1');
            const nested2 = new lib_1.NestedStack(stack, 'Nested2');
            const nested21 = new lib_1.NestedStack(nested2, 'Nested21');
            // WHEN
            nested1.addDependency(nested21);
            // THEN: transfered to a resource dep between the resources in the common stack
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
                DependsOn: [stack.resolve(nested2.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'deeply nested stack depends on a parent nested stack'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const nested1 = new lib_1.NestedStack(stack, 'Nested1');
            const nested2 = new lib_1.NestedStack(stack, 'Nested2');
            const nested21 = new lib_1.NestedStack(nested2, 'Nested21');
            // WHEN
            nested21.addDependency(nested1);
            // THEN: transfered to a resource dep between the resources in the common stack
            assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudFormation::Stack', {
                DependsOn: [stack.resolve(nested1.nestedStackResource.logicalId)],
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'top-level stack depends on a nested stack within a sibling'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack1 = new core_1.Stack(app, 'Stack1');
            const nested1 = new lib_1.NestedStack(stack1, 'Nested1');
            const stack2 = new core_1.Stack(app, 'Stack2');
            // WHEN
            stack2.addDependency(nested1);
            // THEN: assembly-level dependency between stack2 and stack1
            const assembly = app.synth();
            assertAssemblyDependency(test, assembly, stack2, ['Stack1']);
            assertAssemblyDependency(test, assembly, stack1, []);
            assertNoDependsOn(test, assembly, stack1);
            assertNoDependsOn(test, assembly, stack2);
            assertNoDependsOn(test, assembly, nested1);
            test.done();
        },
        'nested stack within a sibling depends on top-level stack'(test) {
            // GIVEN
            const app = new core_1.App();
            const stack1 = new core_1.Stack(app, 'Stack1');
            const nested1 = new lib_1.NestedStack(stack1, 'Nested1');
            const stack2 = new core_1.Stack(app, 'Stack2');
            // WHEN
            nested1.addDependency(stack2);
            // THEN: assembly-level dependency between stack2 and stack1
            const assembly = app.synth();
            assertAssemblyDependency(test, assembly, stack2, []);
            assertAssemblyDependency(test, assembly, stack1, ['Stack2']);
            assertNoDependsOn(test, assembly, stack1);
            assertNoDependsOn(test, assembly, stack2);
            assertNoDependsOn(test, assembly, nested1);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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