"use strict";
const assert_1 = require("@aws-cdk/assert");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type }
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type }
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
function testGrant(test, expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
        "PolicyDocument": {
            "Statement": [
                {
                    "Action": action,
                    "Effect": "Allow",
                    "Resource": [
                        { "Fn::GetAtt": ["mytable0324D45C", "Arn"] },
                        { "Ref": "AWS::NoValue" }
                    ]
                }
            ],
            "Version": "2012-10-17"
        },
        "Users": [{ "Ref": "user2C2B57AE" }]
    }));
    test.done();
}
module.exports = {
    'default properties': {
        'hash key only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
                KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'removalPolicy is DESTROY'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
        'hash + range key'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            }));
            test.done();
        },
        'hash + range key can also be specified in props'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            }));
            test.done();
        },
        'point-in-time recovery is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'server-side encryption is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'stream is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'ttl is not enabled'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
            }));
            test.done();
        },
        'can specify new and old images'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                TableName: 'MyTable'
            }));
            test.done();
        },
        'can specify new images only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.NEW_IMAGE,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
                TableName: 'MyTable',
            }));
            test.done();
        },
        'can specify old images only'(test) {
            const stack = new core_1.Stack();
            new lib_1.Table(stack, CONSTRUCT_NAME, {
                tableName: TABLE_NAME,
                readCapacity: 42,
                writeCapacity: 1337,
                stream: lib_1.StreamViewType.OLD_IMAGE,
                partitionKey: TABLE_PARTITION_KEY,
                sortKey: TABLE_SORT_KEY
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'sortKey', KeyType: 'RANGE' }
                ],
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
                AttributeDefinitions: [
                    { AttributeName: 'hashKey', AttributeType: 'S' },
                    { AttributeName: 'sortKey', AttributeType: 'N' }
                ],
                StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
                TableName: 'MyTable',
            }));
            test.done();
        }
    },
    'when specifying every property'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            pointInTimeRecovery: true,
            serverSideEncryption: true,
            billingMode: lib_1.BillingMode.PROVISIONED,
            stream: lib_1.StreamViewType.KEYS_ONLY,
            timeToLiveAttribute: 'timeToLive',
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: {
                ReadCapacityUnits: 42,
                WriteCapacityUnits: 1337
            },
            PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
            SSESpecification: { SSEEnabled: true },
            StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
            TableName: 'MyTable',
            Tags: [{ Key: 'Environment', Value: 'Production' }],
            TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true }
        }));
        test.done();
    },
    'when specifying PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
            ],
            BillingMode: 'PAY_PER_REQUEST',
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
            ],
            TableName: 'MyTable',
        }));
        test.done();
    },
    'error when specifying read or write capacity with a PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            readCapacity: 1
        }));
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            writeCapacity: 1
        }));
        test.throws(() => new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            readCapacity: 1,
            writeCapacity: 1
        }));
        test.done();
    },
    'when adding a global secondary index with hash key only'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with hash + range key'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.ALL,
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with projection type KEYS_ONLY'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'KEYS_ONLY' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index with projection type INCLUDE'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
            readCapacity: 42,
            writeCapacity: 1337
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
                { AttributeName: 'gsiSortKey', AttributeType: 'B' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                        { AttributeName: 'gsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                    ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 }
                }
            ]
        }));
        test.done();
    },
    'when adding a global secondary index on a table with PAY_PER_REQUEST billing mode'(test) {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY
        }).addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            ],
            BillingMode: 'PAY_PER_REQUEST',
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' }
                }
            ]
        }));
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE
        }), /non-key attributes should be specified when using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type ALL, but with non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value]
        }), /non-key attributes should not be specified when not using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY,
            nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value]
        }), /non-key attributes should not be specified when not using INCLUDE projection type/);
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
        const gsiNonKeyAttributes = [];
        for (let i = 0; i < 21; i++) {
            gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
        }
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: gsiNonKeyAttributes
        }), /a maximum number of nonKeyAttributes across all of secondary indexes is 20/);
        test.done();
    },
    'error when adding a global secondary index with projection type INCLUDE, but with key attributes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [GSI_NON_KEY, TABLE_PARTITION_KEY.name]
            // tslint:disable-next-line:max-line-length
        }), /a key attribute, hashKey, is part of a list of non-key attributes, gsiNonKey,hashKey, which is not allowed since all key attributes are added automatically and this configuration causes stack creation failure/);
        test.done();
    },
    'error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            billingMode: lib_1.BillingMode.PAY_PER_REQUEST
        });
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            readCapacity: 1
        }));
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            writeCapacity: 1
        }));
        test.throws(() => table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
            sortKey: GSI_SORT_KEY,
            readCapacity: 1,
            writeCapacity: 1
        }));
        test.done();
    },
    'when adding multiple global secondary indexes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const gsiGenerator = GSI_GENERATOR();
        for (let i = 0; i < 5; i++) {
            table.addGlobalSecondaryIndex(gsiGenerator.next().value);
        }
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
                { AttributeName: 'gsiHashKey4', AttributeType: 'S' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI0',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI1',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI2',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI3',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
                {
                    IndexName: 'MyGSI4',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                },
            ]
        }));
        test.done();
    },
    'when adding a global secondary index without specifying read and write capacity'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'gsiHashKey', AttributeType: 'S' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            GlobalSecondaryIndexes: [
                {
                    IndexName: 'MyGSI',
                    KeySchema: [
                        { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    ],
                    Projection: { ProjectionType: 'ALL' },
                    ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 }
                }
            ]
        }));
        test.done();
    },
    'when adding a local secondary index with hash + range key'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'ALL' },
                }
            ],
        }));
        test.done();
    },
    'when adding a local secondary index with projection type KEYS_ONLY'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.KEYS_ONLY
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { ProjectionType: 'KEYS_ONLY' },
                }
            ],
        }));
        test.done();
    },
    'when adding a local secondary index with projection type INCLUDE'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY,
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value]
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
                { AttributeName: 'lsiSortKey', AttributeType: 'N' }
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' }
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            LocalSecondaryIndexes: [
                {
                    IndexName: 'MyLSI',
                    KeySchema: [
                        { AttributeName: 'hashKey', KeyType: 'HASH' },
                        { AttributeName: 'lsiSortKey', KeyType: 'RANGE' }
                    ],
                    Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
                }
            ],
        }));
        test.done();
    },
    'error when adding more than 5 local secondary indexes'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        const lsiGenerator = LSI_GENERATOR();
        for (let i = 0; i < 5; i++) {
            table.addLocalSecondaryIndex(lsiGenerator.next().value);
        }
        test.throws(() => table.addLocalSecondaryIndex(lsiGenerator.next().value), /a maximum number of local secondary index per table is 5/);
        test.done();
    },
    'error when adding a local secondary index with the name of a global secondary index'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
        table.addGlobalSecondaryIndex({
            indexName: 'SecondaryIndex',
            partitionKey: GSI_PARTITION_KEY
        });
        test.throws(() => table.addLocalSecondaryIndex({
            indexName: 'SecondaryIndex',
            sortKey: LSI_SORT_KEY
        }), /a duplicate index name, SecondaryIndex, is not allowed/);
        test.done();
    },
    'error when validating construct if a local secondary index exists without a sort key of the table'(test) {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        table.addLocalSecondaryIndex({
            indexName: LSI_NAME,
            sortKey: LSI_SORT_KEY
        });
        const errors = core_1.ConstructNode.validate(table.node);
        test.strictEqual(1, errors.length);
        test.strictEqual('a sort key of the table must be specified to add local secondary indexes', errors[0].message);
        test.done();
    },
    'can enable Read AutoScaling'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // WHEN
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MaxCapacity: 500,
            MinCapacity: 50,
            ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
            ServiceNamespace: 'dynamodb'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
                TargetValue: 75
            }
        }));
        test.done();
    },
    'can enable Write AutoScaling'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // WHEN
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MaxCapacity: 500,
            MinCapacity: 50,
            ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
            ServiceNamespace: 'dynamodb'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
                TargetValue: 75
            }
        }));
        test.done();
    },
    'cannot enable AutoScaling twice on the same property'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
        // WHEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.done();
    },
    'error when enabling AutoScaling on the PAY_PER_REQUEST table'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
        table.addGlobalSecondaryIndex({
            indexName: GSI_NAME,
            partitionKey: GSI_PARTITION_KEY
        });
        // WHEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.throws(() => {
            table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
        });
        test.throws(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
            minCapacity: 1,
            maxCapacity: 5
        }));
        test.done();
    },
    'error when specifying Read Auto Scaling with invalid scalingTargetValue < 10'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // THEN
        test.throws(() => {
            table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
        });
        test.done();
    },
    'error when specifying Read Auto Scaling with invalid minimumCapacity'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
        // THEN
        test.throws(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }));
        test.done();
    },
    'can autoscale on a schedule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
            readCapacity: 42,
            writeCapacity: 1337,
            partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING }
        });
        // WHEN
        const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
        scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
            schedule: appscaling.Schedule.cron({}),
            maxCapacity: 10
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: { "MaxCapacity": 10 },
                    Schedule: "cron(* * * * ? *)",
                    ScheduledActionName: "SaveMoneyByNotScalingUp"
                }
            ]
        }));
        test.done();
    },
    'grants': {
        '"grant" allows adding arbitrary actions associated with this table resource'(test) {
            testGrant(test, ['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
        },
        '"grantReadData" allows the principal to read data from the table'(test) {
            testGrant(test, ['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
        },
        '"grantWriteData" allows the principal to write data to the table'(test) {
            testGrant(test, [
                'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
            ], (p, t) => t.grantWriteData(p));
        },
        '"grantReadWriteData" allows the principal to read/write data'(test) {
            testGrant(test, [
                'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
                'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
            ], (p, t) => t.grantReadWriteData(p));
        },
        '"grantFullAccess" allows the principal to perform any action on the table ("*")'(test) {
            testGrant(test, ['*'], (p, t) => t.grantFullAccess(p));
        },
        '"Table.grantListStreams" allows principal to list all streams'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const user = new iam.User(stack, 'user');
            // WHEN
            lib_1.Table.grantListStreams(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": "*"
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        '"grantTableListStreams" should fail if streaming is not enabled on table"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                }
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            test.throws(() => table.grantTableListStreams(user), /DynamoDB Streams must be enabled on the table my-table/);
            test.done();
        },
        '"grantTableListStreams" allows principal to list all streams for this table'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                stream: lib_1.StreamViewType.NEW_IMAGE
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantTableListStreams(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/stream/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        '"grantStreamRead" should fail if streaming is not enabled on table"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                }
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            test.throws(() => table.grantStreamRead(user), /DynamoDB Streams must be enabled on the table my-table/);
            test.done();
        },
        '"grantStreamRead" allows principal to read and describe the table stream"'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', {
                partitionKey: {
                    name: 'id',
                    type: lib_1.AttributeType.STRING
                },
                stream: lib_1.StreamViewType.NEW_IMAGE
            });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantStreamRead(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "dynamodb:ListStreams",
                            "Effect": "Allow",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/stream/*"]] }
                        },
                        {
                            "Action": [
                                "dynamodb:DescribeStream",
                                "dynamodb:GetRecords",
                                "dynamodb:GetShardIterator"
                            ],
                            "Effect": "Allow",
                            "Resource": {
                                "Fn::GetAtt": [
                                    "mytable0324D45C",
                                    "StreamArn"
                                ]
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        },
        'if table has an index grant gives access to the index'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
            table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
            const user = new iam.User(stack, 'user');
            // WHEN
            table.grantReadData(user);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                { "Fn::GetAtt": ["mytable0324D45C", "Arn"] },
                                { "Fn::Join": ["", [{ "Fn::GetAtt": ["mytable0324D45C", "Arn"] }, "/index/*"]] }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "Users": [{ "Ref": "user2C2B57AE" }]
            }));
            test.done();
        }
    },
    'import': {
        'report error when importing an external/existing table from invalid arn missing resource name'(test) {
            const stack = new core_1.Stack();
            const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
            // WHEN
            test.throws(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn), /ARN for DynamoDB table must be in the form: .../);
            test.done();
        },
        'static import(ref) allows importing an external/existing table from arn'(test) {
            const stack = new core_1.Stack();
            const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
            const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            table.grantReadData(role);
            // it is possible to obtain a permission statement for a ref
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                tableArn,
                                { "Ref": "AWS::NoValue" }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "PolicyName": 'NewRoleDefaultPolicy90E8F49D',
                "Roles": [{ "Ref": 'NewRole99763075' }]
            }));
            test.deepEqual(table.tableArn, tableArn);
            test.deepEqual(stack.resolve(table.tableName), 'MyTable');
            test.done();
        },
        'static import(ref) allows importing an external/existing table from table name'(test) {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            table.grantReadWriteData(role);
            // it is possible to obtain a permission statement for a ref
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan',
                                'dynamodb:BatchWriteItem',
                                'dynamodb:PutItem',
                                'dynamodb:UpdateItem',
                                'dynamodb:DeleteItem'
                            ],
                            "Effect": "Allow",
                            "Resource": [
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                "Ref": "AWS::Partition"
                                            },
                                            ":dynamodb:",
                                            {
                                                "Ref": "AWS::Region"
                                            },
                                            ":",
                                            {
                                                "Ref": "AWS::AccountId"
                                            },
                                            ":table/MyTable"
                                        ]
                                    ]
                                },
                                {
                                    "Ref": "AWS::NoValue"
                                }
                            ]
                        }
                    ],
                    "Version": "2012-10-17"
                },
                "PolicyName": 'NewRoleDefaultPolicy90E8F49D',
                "Roles": [{ "Ref": 'NewRole99763075' }]
            }));
            test.deepEqual(table.tableArn, 'arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
            test.deepEqual(stack.resolve(table.tableName), tableName);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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