"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @stability stable
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        /** Set to keep track of what target accounts and regions we've already created event buses for */
        this._xEnvTargetsAdded = new Set();
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @stability stable
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @stability stable
     */
    addTarget(target) {
        var _b, _c, _d;
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = ((_c = targetProps.targetResource.env) === null || _c === void 0 ? void 0 : _c.account) || targetStack.account;
            const targetRegion = ((_d = targetProps.targetResource.env) === null || _d === void 0 ? void 0 : _d.region) || targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            // if the target is in a different account or region and is defined in this CDK App
            // we can generate all the needed components:
            // - forwarding rule in the source stack (target: default event bus of the receiver region)
            // - eventbus permissions policy (creating an extra stack)
            // - receiver rule in the target stack (target: the actual target)
            if (!util_1.sameEnvDimension(sourceAccount, targetAccount) || !util_1.sameEnvDimension(sourceRegion, targetRegion)) {
                // cross-account and/or cross-region event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cross-account.html
                // for cross-account or cross-region events, we require a concrete target account and region
                if (!targetAccount || core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account or cross-region events');
                }
                if (!targetRegion || core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account or cross-region events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account or cross-region events');
                }
                // Don't exactly understand why this code was here (seems unlikely this rule would be violated), but
                // let's leave it in nonetheless.
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account or cross-region targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account or cross-region event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                // The target of this Rule will be the default event bus of the target environment
                this.ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id);
                // The actual rule lives in the target stack. Other than the account, it's identical to this one,
                // but only evaluated at render time (via a special subclass).
                //
                // FIXME: the MirrorRule is a bit silly, forwarding the exact same event to another event bus
                // and trigger on it there (there will be issues with construct references, for example). Especially
                // in the case of scheduled events, we will just trigger both rules in parallel in both environments.
                //
                // A better solution would be to have the source rule add a unique token to the the event,
                // and have the mirror rule trigger on that token only (thereby properly separating triggering, which
                // happens in the source env; and activating, which happens in the target env).
                //
                // Don't have time to do that right now.
                const mirrorRuleScope = this.obtainMirrorRuleScope(targetStack, targetAccount, targetRegion);
                new MirrorRule(mirrorRuleScope, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                }, this);
                return;
            }
        }
        // Here only if the target does not have a targetResource defined.
        // In such case we don't have to generate any extra component.
        // Note that this can also be an imported resource (i.e: EventBus target)
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            httpParameters: targetProps.httpParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @stability stable
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        return util_1.renderEventPattern(this.eventPattern);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    /**
     * Make sure we add the target environments event bus as a target, and the target has permissions set up to receive our events
     *
     * For cross-account rules, uses a support stack to set up a policy on the target event bus.
     */
    ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id) {
        // the _actual_ target is just the event bus of the target's account
        // make sure we only add it once per account per region
        const key = `${targetAccount}:${targetRegion}`;
        if (this._xEnvTargetsAdded.has(key)) {
            return;
        }
        this._xEnvTargetsAdded.add(key);
        const eventBusArn = targetStack.formatArn({
            service: 'events',
            resource: 'event-bus',
            resourceName: 'default',
            region: targetRegion,
            account: targetAccount,
        });
        // For some reason, cross-region requires a Role (with `PutEvents` on the
        // target event bus) while cross-account doesn't
        const roleArn = !util_1.sameEnvDimension(targetRegion, core_1.Stack.of(this).region)
            ? this.crossRegionPutEventsRole(eventBusArn).roleArn
            : undefined;
        this.targets.push({
            id,
            arn: eventBusArn,
            roleArn,
        });
        // Add a policy to the target Event Bus to allow the source account/region to publish into it.
        //
        // Since this Event Bus permission needs to be deployed before the stack containing the Rule is deployed
        // (as EventBridge verifies whether you have permissions to the targets on rule creation), this needs
        // to be in a support stack.
        const sourceApp = this.node.root;
        const sourceAccount = core_1.Stack.of(this).account;
        // If different accounts, we need to add the permissions to the target eventbus
        //
        // For different region, no need for a policy on the target event bus (but a need
        // for a role).
        if (!util_1.sameEnvDimension(sourceAccount, targetAccount)) {
            const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
            let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
            if (!eventBusPolicyStack) {
                eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                    env: {
                        account: targetAccount,
                        region: targetRegion,
                    },
                    // The region in the stack name is rather redundant (it will always be the target region)
                    // Leaving it in for backwards compatibility.
                    stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                });
                new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                    action: 'events:PutEvents',
                    statementId: `Allow-account-${sourceAccount}`,
                    principal: sourceAccount,
                });
            }
            // deploy the event bus permissions before the source stack
            core_1.Stack.of(this).addDependency(eventBusPolicyStack);
        }
    }
    /**
     * Return the scope where the mirror rule should be created for x-env event targets
     *
     * This is the target resource's containing stack if it shares the same region (owned
     * resources), or should be a fresh support stack for imported resources.
     *
     * We don't implement the second yet, as I have to think long and hard on whether we
     * can reuse the existing support stack or not, and I don't have time for that right now.
     */
    obtainMirrorRuleScope(targetStack, targetAccount, targetRegion) {
        // for cross-account or cross-region events, we cannot create new components for an imported resource
        // because we don't have the target stack
        if (util_1.sameEnvDimension(targetStack.account, targetAccount) && util_1.sameEnvDimension(targetStack.region, targetRegion)) {
            return targetStack;
        }
        // For now, we don't do the work for the support stack yet
        throw new Error('Cannot create a cross-account or cross-region rule for an imported resource (create a stack with the right environment for the imported resource)');
    }
    /**
     * Obtain the Role for the EventBridge event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     * @internal
     */
    crossRegionPutEventsRole(eventBusArn) {
        const id = 'EventsRole';
        let role = this.node.tryFindChild(id);
        if (!role) {
            role = new aws_iam_1.Role(this, id, {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
            });
        }
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [eventBusArn],
        }));
        return role;
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "@aws-cdk/aws-events.Rule", version: "1.132.0" };
/**
 * A rule that mirrors another rule
 */
class MirrorRule extends Rule {
    constructor(scope, id, props, source) {
        super(scope, id, props);
        this.source = source;
    }
    _renderEventPattern() {
        return this.source._renderEventPattern();
    }
    /**
     * Override validate to be a no-op
     *
     * The rules are never stored on this object so there's nothing to validate.
     *
     * Instead, we mirror the other rule at render time.
     */
    validate() {
        return [];
    }
}
//# sourceMappingURL=data:application/json;base64,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