"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const eks = require("@aws-cdk/aws-eks");
const user_data_1 = require("./user-data");
const iam = require("@aws-cdk/aws-iam");
const ec2 = require("@aws-cdk/aws-ec2");
const launch_template_1 = require("./launch-template");
const DEFAULT_INSTANCE_TYPE = 't3.large';
var BlockDuration;
(function (BlockDuration) {
    BlockDuration[BlockDuration["ONE_HOUR"] = 60] = "ONE_HOUR";
    BlockDuration[BlockDuration["TWO_HOURS"] = 120] = "TWO_HOURS";
    BlockDuration[BlockDuration["THREE_HOURS"] = 180] = "THREE_HOURS";
    BlockDuration[BlockDuration["FOUR_HOURS"] = 240] = "FOUR_HOURS";
    BlockDuration[BlockDuration["FIVE_HOURS"] = 300] = "FIVE_HOURS";
    BlockDuration[BlockDuration["SIX_HOURS"] = 360] = "SIX_HOURS";
})(BlockDuration = exports.BlockDuration || (exports.BlockDuration = {}));
var InstanceInterruptionBehavior;
(function (InstanceInterruptionBehavior) {
    InstanceInterruptionBehavior["HIBERNATE"] = "hibernate";
    InstanceInterruptionBehavior["STOP"] = "stop";
    InstanceInterruptionBehavior["TERMINATE"] = "terminate";
})(InstanceInterruptionBehavior = exports.InstanceInterruptionBehavior || (exports.InstanceInterruptionBehavior = {}));
var ClusterVersion;
(function (ClusterVersion) {
    ClusterVersion["KUBERNETES_114"] = "1.14";
    ClusterVersion["KUBERNETES_115"] = "1.15";
    ClusterVersion["KUBERNETES_116"] = "1.16";
})(ClusterVersion = exports.ClusterVersion || (exports.ClusterVersion = {}));
class EksSpotCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        // this.cluster = props.cluster;
        this.clusterVersion = props.clusterVersion;
        // use an existing vpc or create a new one
        this.vpc = this.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(this, 'Vpc', { isDefault: true }) :
            this.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(this, 'Vpc', { vpcId: this.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        const clusterAdmin = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        this.cluster = new eks.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            mastersRole: clusterAdmin,
            defaultCapacity: 0,
            version: this.clusterVersion,
        });
    }
    addSpotFleet(id, props) {
        new SpotFleet(this, id, {
            cluster: this,
            ...props,
        });
    }
    addDays(date, days) {
        date.setDate(date.getDate() + days);
        return date;
    }
    addHours(date, hours) {
        date.setHours(date.getHours() + hours);
        return date;
    }
    addMinutes(date, minutes) {
        date.setMinutes(date.getMinutes() + minutes);
        return date;
    }
}
exports.EksSpotCluster = EksSpotCluster;
class SpotFleet extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id, props);
        this.spotFleetId = id;
        this.clusterStack = props.cluster;
        this.launchTemplate = (_a = props.launchTemplate) !== null && _a !== void 0 ? _a : new launch_template_1.LaunchTemplate();
        this.targetCapacity = props.targetCapacity;
        this.defaultInstanceType = (_b = props.defaultInstanceType) !== null && _b !== void 0 ? _b : new ec2.InstanceType(DEFAULT_INSTANCE_TYPE);
        // isntance role
        this.instanceRole = props.instanceRole || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        this.instanceRole.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const sg = new ec2.SecurityGroup(this, 'SpotFleetSg', {
            vpc: this.clusterStack.cluster.vpc,
        });
        // self rules
        sg.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        sg.connections.allowFrom(this.clusterStack.cluster, ec2.Port.tcp(443));
        sg.connections.allowFrom(this.clusterStack.cluster, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        sg.connections.allowTo(this.clusterStack.cluster, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        sg.connections.allowToAnyIpv4(ec2.Port.allTcp());
        sg.connections.allowToAnyIpv4(ec2.Port.allUdp());
        sg.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const config = this.launchTemplate.bind(this);
        // const userData = renderAmazonLinuxUserData(cdk.Stack.of(this), this.cluster.clusterName, config.spotfleet);
        const userData = ec2.UserData.forLinux();
        userData.addCommands(...user_data_1.renderAmazonLinuxUserData(core_1.Stack.of(this), this.clusterStack.cluster.clusterName, config.spotfleet));
        this.defaultInstanceType = (_c = props.defaultInstanceType) !== null && _c !== void 0 ? _c : new ec2.InstanceType(DEFAULT_INSTANCE_TYPE);
        const imageId = (_d = props.customAmiId) !== null && _d !== void 0 ? _d : new eks.EksOptimizedImage({
            nodeType: nodeTypeForInstanceType(this.defaultInstanceType),
            kubernetesVersion: props.cluster.clusterVersion,
        }).getImage(this).imageId;
        const lt = new ec2.CfnLaunchTemplate(this, 'LaunchTemplate', {
            launchTemplateData: {
                imageId,
                instanceType: this.defaultInstanceType.toString(),
                tagSpecifications: [
                    {
                        resourceType: 'instance',
                        tags: [
                            {
                                key: 'Name',
                                value: `${core_1.Stack.of(this).stackName}/Cluster/${this.spotFleetId}`,
                            },
                            {
                                key: `kubernetes.io/cluster/${this.clusterStack.cluster.clusterName}`,
                                value: 'owned',
                            },
                        ],
                    },
                ],
                instanceMarketOptions: {
                    marketType: 'spot',
                    spotOptions: {
                        blockDurationMinutes: (_e = props.blockDuration) !== null && _e !== void 0 ? _e : BlockDuration.ONE_HOUR,
                        instanceInterruptionBehavior: (_f = props.instanceInterruptionBehavior) !== null && _f !== void 0 ? _f : InstanceInterruptionBehavior.TERMINATE,
                    },
                },
                // userData: cdk.Fn.base64(capacityAsg.userData.render()),
                userData: core_1.Fn.base64(userData.render()),
                securityGroupIds: sg.connections.securityGroups.map(m => m.securityGroupId),
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
            },
        });
        const spotFleetRole = new iam.Role(this, 'FleetRole', {
            assumedBy: new iam.ServicePrincipal('spotfleet.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
            ],
        });
        const overrides = [];
        for (const s of this.clusterStack.cluster.vpc.privateSubnets) {
            overrides.push({ subnetId: s.subnetId });
        }
        new ec2.CfnSpotFleet(this, id, {
            spotFleetRequestConfigData: {
                launchTemplateConfigs: [
                    {
                        launchTemplateSpecification: {
                            launchTemplateId: lt.ref,
                            version: lt.attrLatestVersionNumber,
                        },
                        overrides,
                    },
                ],
                iamFleetRole: spotFleetRole.roleArn,
                targetCapacity: (_g = props.targetCapacity) !== null && _g !== void 0 ? _g : 1,
                validFrom: props.validFrom,
                validUntil: props.validUntil,
                terminateInstancesWithExpiration: props.terminateInstancesWithExpiration,
            },
        });
        this.clusterStack.cluster.awsAuth.addRoleMapping(this.instanceRole, {
            username: 'system:node:{{EC2PrivateDNSName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
            ],
        });
    }
}
exports.SpotFleet = SpotFleet;
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? eks.NodeType.GPU : eks.NodeType.STANDARD;
}
//# sourceMappingURL=data:application/json;base64,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