"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionType = exports.PassthroughBehavior = exports.IntegrationType = exports.ContentHandling = exports.Integration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const vpc_link_1 = require("./vpc-link");
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IntegrationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Integration);
            }
            throw error;
        }
        const options = this.props.options || {};
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.timeout && !options.timeout.isUnresolved() && (options.timeout.toMilliseconds() < 50 || options.timeout.toMilliseconds() > 29000)) {
            throw new Error('Integration timeout must be between 50 milliseconds and 29 seconds.');
        }
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_Method(_method);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        let uri = this.props.uri;
        const options = this.props.options;
        if (options?.connectionType === ConnectionType.VPC_LINK && uri === undefined) {
            uri = core_1.Lazy.string({
                // needs to be a lazy since the targets can be added to the VpcLink construct after initialization.
                produce: () => {
                    const vpcLink = options.vpcLink;
                    if (vpcLink instanceof vpc_link_1.VpcLink) {
                        const targets = vpcLink._targetDnsNames;
                        if (targets.length > 1) {
                            throw new Error("'uri' is required when there are more than one NLBs in the VPC Link");
                        }
                        else {
                            return `http://${targets[0]}`;
                        }
                    }
                    else {
                        throw new Error("'uri' is required when the 'connectionType' is VPC_LINK");
                    }
                },
            });
        }
        return {
            options: {
                ...options,
                connectionType: options?.vpcLink ? ConnectionType.VPC_LINK : options?.connectionType,
            },
            type: this.props.type,
            uri,
            integrationHttpMethod: this.props.integrationHttpMethod,
        };
    }
}
exports.Integration = Integration;
_a = JSII_RTTI_SYMBOL_1;
Integration[_a] = { fqn: "@aws-cdk/aws-apigateway.Integration", version: "1.164.0" };
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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