import contextlib
import io
import sys

import pytest
import numpy as np

import psyneulink as pnl
from psyneulink.core.compositions.report import ReportOutput, ReportProgress, ReportSimulations, ReportDevices


@pytest.mark.skipif(sys.platform == 'win32', reason="<Incompatible UDF-8 formatting of rich Console output>")
class TestReport():

    def test_reportOutputPref_true(self):

        t = pnl.TransferMechanism()
        t.reportOutputPref = ReportOutput.FULL

        f = io.StringIO()
        with contextlib.redirect_stdout(f):
            t.execute(1)
        output = f.getvalue()

        assert 'input: 1.0' in output
        assert 'output: 1.0' in output
        assert 'params' not in output

    def test_reportOutputPref_params(self):

        t = pnl.TransferMechanism()
        t.reportOutputPref = 'params'

        f = io.StringIO()
        with contextlib.redirect_stdout(f):
            t.execute(1, report_output=ReportOutput.FULL)
        output = f.getvalue()

        assert 'input: 1.0' in output
        assert 'output: 1.0' in output
        assert 'params' in output

        # NOTE: parameters are not consistent in printed form with
        # their underlying values (e.g. dimension brackets are removed)
        # So, don't check output for all parameters and correct values
        assert 'noise:' in output
        assert 'integration_rate:' in output

        assert 'Parameter(' not in output
        assert 'pnl_internal=' not in output

    def test_simple_output_and_progress(self):
        """Test simple sequence of three Mechanisms, using all report_output and report_progress options
        """

        a = pnl.TransferMechanism(name='a')
        b = pnl.TransferMechanism(name='b')
        c = pnl.TransferMechanism(name='c')
        comp = pnl.Composition(pathways=[a,b,c], name='COMP')

        a.reportOutputPref=ReportOutput.OFF
        b.reportOutputPref=ReportOutput.ON
        c.reportOutputPref=ReportOutput.FULL

        comp.run(report_output=ReportOutput.TERSE,
                 report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 0 ====================\\n    Time Step 0 ---------\\n      a executed\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n      c executed\\n\''
        assert repr(actual_output) == expected_output

        # Test Composition.run()
        comp.run(report_output=ReportOutput.TERSE,
                 report_progress=ReportProgress.ON,
                 report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 0 ====================\\n    Time Step 0 ---------\\n      a executed\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n      c executed\\nCOMP: Executed 1 of 1 trial\\n\''
        assert repr(actual_output) == expected_output

        comp.run(report_output=ReportOutput.USE_PREFS, report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 0 ====================\\n    Time Step 0 ---------\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n╭───── c ─────╮\\n│ input: 0.0  │\\n│ output: 0.0 │\\n╰─────────────╯\\n\''
        assert repr(actual_output) == expected_output

        comp.run(report_output=ReportOutput.USE_PREFS,
                 report_progress=ReportProgress.ON,
                 report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\nExecution of COMP:\n    COMP TRIAL 0 ====================\n    Time Step 0 ---------\n    Time Step 1 ---------\n      b executed\n    Time Step 2 ---------\n╭───── c ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\nCOMP: Executed 1 of 1 trial\n'
        assert actual_output == expected_output

        comp.run(report_output=ReportOutput.FULL, report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\n ╔═══ EXECUTION OF COMP  ════╗\n ║                           ║\n ║                           ║\n ║  ┏━━  COMP: Trial 0  ━━━┓ ║\n ║  ┃                      ┃ ║\n ║  ┃  input: [[0.0]]      ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 0 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── a ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 1 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── b ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── c ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  result: [[0.0]]     ┃ ║\n ║  ┃                      ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                           ║\n ╚═══════════════════════════╝\n\n'
        assert actual_output == expected_output

        comp.run(report_output=ReportOutput.FULL,
                 report_progress=ReportProgress.ON,
                 report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\n ╔═══ EXECUTION OF COMP  ════╗\n ║                           ║\n ║                           ║\n ║  ┏━━  COMP: Trial 0  ━━━┓ ║\n ║  ┃                      ┃ ║\n ║  ┃  input: [[0.0]]      ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 0 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── a ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 1 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── b ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── c ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  result: [[0.0]]     ┃ ║\n ║  ┃                      ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                           ║\n ╚═══════════════════════════╝\n\nCOMP: Executed 1 of 1 trial\n'
        assert actual_output == expected_output

        # Run these tests after ones calling run() above to avoid having to reset trial counter,
        # which increments after calls to execute()
        comp.execute(report_output=ReportOutput.TERSE,
                     report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 0 ====================\\n    Time Step 0 ---------\\n      a executed\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n      c executed\\n\''
        assert repr(actual_output) == expected_output

        # Test Composition.execute()
        comp.execute(report_output=ReportOutput.TERSE,
                     report_progress=ReportProgress.ON,
                     report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 1 ====================\\n    Time Step 0 ---------\\n      a executed\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n      c executed\\nCOMP: Executed 1 of 1 trial\\n\''
        assert repr(actual_output) == expected_output

        comp.execute(report_output=ReportOutput.USE_PREFS, report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\'\\nExecution of COMP:\\n    COMP TRIAL 2 ====================\\n    Time Step 0 ---------\\n    Time Step 1 ---------\\n      b executed\\n    Time Step 2 ---------\\n╭───── c ─────╮\\n│ input: 0.0  │\\n│ output: 0.0 │\\n╰─────────────╯\\n\''
        assert repr(actual_output) == expected_output

        comp.execute(report_output=ReportOutput.USE_PREFS,
                     report_progress=ReportProgress.ON,
                     report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\nExecution of COMP:\n    COMP TRIAL 3 ====================\n    Time Step 0 ---------\n    Time Step 1 ---------\n      b executed\n    Time Step 2 ---------\n╭───── c ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\nCOMP: Executed 1 of 1 trial\n'
        assert actual_output == expected_output

        comp.execute(report_output=ReportOutput.FULL, report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\n ╔═══ EXECUTION OF COMP  ════╗\n ║                           ║\n ║                           ║\n ║  ┏━━  COMP: Trial 4  ━━━┓ ║\n ║  ┃                      ┃ ║\n ║  ┃  input: [[0.0]]      ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 0 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── a ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 1 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── b ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── c ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  result: [[0.0]]     ┃ ║\n ║  ┃                      ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                           ║\n ╚═══════════════════════════╝\n\n'
        assert actual_output == expected_output

        comp.execute(report_output=ReportOutput.FULL,
                     report_progress=ReportProgress.ON,
                     report_to_devices=ReportDevices.DIVERT)
        actual_output = comp.rich_diverted_reports
        expected_output = '\n ╔═══ EXECUTION OF COMP  ════╗\n ║                           ║\n ║                           ║\n ║  ┏━━  COMP: Trial 5  ━━━┓ ║\n ║  ┃                      ┃ ║\n ║  ┃  input: [[0.0]]      ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 0 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── a ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 1 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── b ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  │ ╭───── c ─────╮ │ ┃ ║\n ║  ┃  │ │ input: 0.0  │ │ ┃ ║\n ║  ┃  │ │ output: 0.0 │ │ ┃ ║\n ║  ┃  │ ╰─────────────╯ │ ┃ ║\n ║  ┃  │                 │ ┃ ║\n ║  ┃  └─────────────────┘ ┃ ║\n ║  ┃                      ┃ ║\n ║  ┃  result: [[0.0]]     ┃ ║\n ║  ┃                      ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                           ║\n ╚═══════════════════════════╝\n\nCOMP: Executed 1 of 1 trial\n'
        assert actual_output == expected_output

    # def test_two_mechs_in_a_time_step(self):
    #     """Test that includes two (recurrently connected) Mechanisms executed within the same TIME_STEP
    #        FIX: NEED TO RESOLVE INDETERMINACY OF ORDER OF EXECUTION FOR TESTING
    #     """
    #     a = TransferMechanism(name='a')
    #     b = TransferMechanism(name='b')
    #     c = TransferMechanism(name='c')
    #     comp = Composition(pathways=[[a,b],[b,a], [a,c]], name='COMP')
    #
    #     a.reportOutputPref=True
    #     b.reportOutputPref=False
    #     c.reportOutputPref=True
    #
    #     comp.run(report_output=TERSE, report_progress=[False, ReportDevices.DIVERT])
    #     actual_output = comp.rich_diverted_reports
    #     expected_output = '\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n  a executed\n  b executed\n Time Step 1 ---------\n  c executed\n\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n  a executed\n  b executed\n Time Step 1 ---------\n  c executed\nCOMP TRIAL 1  ====================\n Time Step 0 ---------\n  a executed\n  b executed\n Time Step 1 ---------\n  c executed\n'
    #     assert actual_output == expected_output
    #
    #     comp.run(report_output=TERSE, report_progress=ReportDevices.DIVERT)
    #     actual_output = comp.rich_diverted_reports
    #     expected_output = '\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n  b executed\n  a executed\n Time Step 1 ---------\n  c executed\nCOMP: Executed 1 of 2 trials\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n  b executed\n  a executed\n Time Step 1 ---------\n  c executed\nCOMP TRIAL 1  ====================\n Time Step 0 ---------\n  b executed\n  a executed\n Time Step 1 ---------\n  c executed\nCOMP: Executed 2 of 2 trials'
    #     assert actual_output == expected_output
    #
    #     comp.run(report_output=True, report_progress=ReportDevices.DIVERT)
    #     actual_output = comp.rich_diverted_reports
    #     expected_output = '\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n╭───── a ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\n Time Step 1 ---------\n╭───── c ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\nCOMP: Executed 1 of 2 trials\nCOMP TRIAL 0  ====================\n Time Step 0 ---------\n╭───── a ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\n Time Step 1 ---------\n╭───── c ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\nCOMP TRIAL 1  ====================\n Time Step 0 ---------\n╭───── a ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\n Time Step 1 ---------\n╭───── c ─────╮\n│ input: 0.0  │\n│ output: 0.0 │\n╰─────────────╯\nCOMP: Executed 2 of 2 trials'
    #     assert actual_output == expected_output
    #
    #     comp.run(num_trials=2, report_output=FULL, report_progress=ReportDevices.DIVERT)
    #     actual_output = comp.rich_diverted_reports
    #     expected_output = '\n┏━━━  COMP: Trial 0  ━━━┓\n┃                       ┃\n┃ input: [[0.0], [0.0]] ┃\n┃                       ┃\n┃ ┌─  Time Step 0 ──┐   ┃\n┃ │ ╭───── b ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ │ ╭───── a ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ ┌─  Time Step 1 ──┐   ┃\n┃ │ ╭───── c ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ result: [[0.0]]       ┃\n┃                       ┃\n┗━━━━━━━━━━━━━━━━━━━━━━━┛\n\nCOMP: Executed 1 of 2 trials\n┏━━━  COMP: Trial 0  ━━━┓\n┃                       ┃\n┃ input: [[0.0], [0.0]] ┃\n┃                       ┃\n┃ ┌─  Time Step 0 ──┐   ┃\n┃ │ ╭───── b ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ │ ╭───── a ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ ┌─  Time Step 1 ──┐   ┃\n┃ │ ╭───── c ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ result: [[0.0]]       ┃\n┃                       ┃\n┗━━━━━━━━━━━━━━━━━━━━━━━┛\n\n┏━━━  COMP: Trial 1  ━━━┓\n┃                       ┃\n┃ input: [[0.0], [0.0]] ┃\n┃                       ┃\n┃ ┌─  Time Step 0 ──┐   ┃\n┃ │ ╭───── b ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ │ ╭───── a ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ ┌─  Time Step 1 ──┐   ┃\n┃ │ ╭───── c ─────╮ │   ┃\n┃ │ │ input: 0.0  │ │   ┃\n┃ │ │ output: 0.0 │ │   ┃\n┃ │ ╰─────────────╯ │   ┃\n┃ └─────────────────┘   ┃\n┃                       ┃\n┃ result: [[0.0]]       ┃\n┃                       ┃\n┗━━━━━━━━━━━━━━━━━━━━━━━┛\n\nCOMP: Executed 2 of 2 trials'
    #     assert actual_output == expected_output

    # FIX: SOME OF THESE MAY NEED TO BE MODIFIED AFTER A BUG IS FIXED IN WHICH A NESTED COMPOSITION CONTINUES TO
    #      INCREMENT ITS TRIAL COUNT OVER SEQUENTIAL EXECUTIONS OF THE OUTER COMPOSITION
    #      (SINCE IT IS CALLED USING execute() RATHER run(), THAT WOULD RESET ITS COUNT ON EACH CALL).
    def test_nested_comps_and_sims_basic(self):
        """Test output and progress reports for execution of simulations by controller in both an outer and nested
        composition, using input dictionary, generator instance and generator function.
        """

        # instantiate mechanisms and inner comp
        ia = pnl.TransferMechanism(name='ia')
        ib = pnl.TransferMechanism(name='ib')
        ia.reportOutputPref=[pnl.INTERCEPT]
        icomp = pnl.Composition(name='icomp', controller_mode=pnl.AFTER)

        # set up structure of inner comp
        icomp.add_node(ia, required_roles=pnl.NodeRole.INPUT)
        icomp.add_node(ib, required_roles=pnl.NodeRole.OUTPUT)
        icomp.add_projection(pnl.MappingProjection(), sender=ia, receiver=ib)

        # add controller to inner comp
        icomp.add_controller(
                pnl.OptimizationControlMechanism(
                        agent_rep=icomp,
                        state_features=[ia.input_port],
                        name="iController",
                        objective_mechanism=pnl.ObjectiveMechanism(
                                monitor=ib.output_port,
                                function=pnl.SimpleIntegrator,
                                name="iController Objective Mechanism"
                        ),
                        function=pnl.GridSearch(direction=pnl.MAXIMIZE),
                        control_signals=[pnl.ControlSignal(projections=[(pnl.SLOPE, ia)],
                                                           variable=1.0,
                                                           intensity_cost_function=pnl.Linear(slope=0.0),
                                                           allocation_samples=pnl.SampleSpec(start=1.0,
                                                                                             stop=10.0,
                                                                                             num=3))])
        )

        # instantiate outer comp
        ocomp = pnl.Composition(name='ocomp', controller_mode=pnl.BEFORE)

        # setup structure of outer comp
        ocomp.add_node(icomp)

        # ocomp._analyze_graph()

        # add controller to outer comp
        ocomp.add_controller(
                pnl.OptimizationControlMechanism(
                        agent_rep=ocomp,
                        state_features=[ia.input_port],
                        # state_features=[ocomp.input_CIM.output_ports[0]],
                        name="oController",
                        objective_mechanism=pnl.ObjectiveMechanism(
                                monitor=ib.output_port,
                                function=pnl.SimpleIntegrator,
                                name="oController Objective Mechanism"
                        ),
                        function=pnl.GridSearch(direction=pnl.MAXIMIZE),
                        control_signals=[pnl.ControlSignal(projections=[(pnl.SLOPE, ia)],
                                                           variable=1.0,
                                                           intensity_cost_function=pnl.Linear(slope=0.0),
                                                           allocation_samples=pnl.SampleSpec(start=1.0,
                                                                                             stop=10.0,
                                                                                             num=2))])
            )

        # set up input using three different formats:
        #  1) generator function
        #  2) instance of generator function
        #  3) inputs dict
        inputs_dict = {
            icomp:
                {
                    ia: [[-2], [1]]
                }
        }

        def inputs_generator_function():
            for i in range(2):
                yield {
                    icomp:
                        {
                            ia: inputs_dict[icomp][ia][i]
                        }
                }

        inputs_generator_instance = inputs_generator_function()

        ib.reportOutputPref=[pnl.VALUE, pnl.VARIABLE]
        icomp.controller.reportOutputPref = ReportOutput.ON
        ocomp.controller.reportOutputPref = ReportOutput.FULL

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 0 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 0\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.USE_PREFS,
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  ╭── oController ───╮\n  │ input: 0.0, -2.0 │\n  │ output: 1.0      │\n  ╰──────────────────╯\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 1 ====================\n      Time Step 0 ---------\n      ╭────────────── ia ───────────────╮\n      │ input: -2.0                     │\n      │ ╭────────── params ───────────╮ │\n      │ │ function: Linear Function-5 │ │\n      │ │     intercept: 0.0          │ │\n      │ ╰─────────────────────────────╯ │\n      │ output: -2.0                    │\n      ╰─────────────────────────────────╯\n      Time Step 1 ---------\n      ╭────────────────────────────────── ib ──────────────────────────────────╮\n      │ input: -2.0                                                            │\n      │ ╭────────────────────────────── params ──────────────────────────────╮ │\n      │ │ value: -2. (monitored by iController Objective Mechanism and       │ │\n      │ │ oController Objective Mechanism)                                   │ │\n      │ │ function: Linear Function-14                                       │ │\n      │ │     variable: -2.0                                                 │ │\n      │ ╰────────────────────────────────────────────────────────────────────╯ │\n      │ output: -2.0                                                           │\n      ╰────────────────────────────────────────────────────────────────────────╯\n      Time Step 2 ---------\n    iController simulation of icomp after TRIAL 1\n    Time Step 1 ---------\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.OFF,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\'ocomp: Executed 1 of 1 trial\\n    icomp: Executed 1 of 1 trial (depth: 1)\\n\''
        assert repr(actual_output) == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 3 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 3\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 1 of 1 trial\n    icomp: Executed 1 of 1 trial (depth: 1)\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 4 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 4\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.OFF,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = 'ocomp: Executed 1 of 1 trial\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 6 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 6\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp before TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 7 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 7\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 2 of 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        ocomp.run(inputs=inputs_generator_instance,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 8 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 8\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp before TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 9 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 9\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        ocomp.run(inputs=inputs_generator_function(),
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 10 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 10\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp before TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 11 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 11\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        # Test other combinations of controller_modes for outer and inner Compositions:
        icomp.controller_mode = pnl.AFTER
        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 12 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 12\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp before TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 13 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 13\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 2 of 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        icomp.controller_mode = pnl.BEFORE
        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    iController simulation of icomp before TRIAL 14\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Execution of icomp within ocomp:\n      icomp TRIAL 14 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp before TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    iController simulation of icomp before TRIAL 15\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Execution of icomp within ocomp:\n      icomp TRIAL 15 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\nocomp: Executed 2 of 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

        ocomp.controller_mode = pnl.AFTER
        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\nExecution of ocomp:\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    iController simulation of icomp before TRIAL 16\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Execution of icomp within ocomp:\n      icomp TRIAL 16 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp after TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 1 ====================\n    Time Step 0 ---------\n    iController simulation of icomp before TRIAL 17\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 2 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Execution of icomp within ocomp:\n      icomp TRIAL 17 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n  oController simulation of ocomp after TRIAL 1\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      iController simulation of icomp before TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 2 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\nocomp: Executed 2 of 2 trials\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n'
        assert actual_output == expected_output

        # Test ReportOutput.FULL

        icomp.controller_mode = pnl.BEFORE
        ocomp.controller_mode = pnl.AFTER

        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp BEFORE its Trial 18 ━┓  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                       ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [54.0]                                       ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                           ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 18  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                     │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                   │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -20. (monitored by iController      │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -20.0                        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                                  │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 34.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-20.0]]                                      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: -20.0                      │ │                             ┃ ║\n ║  ┃  │ │ output: 34.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-20.0]]                                                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━  oController SIMULATION OF ocomp AFTER its Trial 0 ━┓                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┃  state: [[-2.0]]                                     ┃                   ║\n ║  ┃  outcome: [34.0]                                     ┃                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┃  control allocation: [[1.0]]                         ┃                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛                   ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 1  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1]]                                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp BEFORE its Trial 19 ━┓  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[1.0]]                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [34.0]                                       ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[10.0]]                          ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                        ┃  ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 19  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 1.0                      │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                    │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: 10. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: 10.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 44.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[10.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: 10.0                       │ │                             ┃ ║\n ║  ┃  │ │ output: 44.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[10.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━  oController SIMULATION OF ocomp AFTER its Trial 1 ━┓                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┃  state: [[1.0]]                                      ┃                   ║\n ║  ┃  outcome: [44.0]                                     ┃                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┃  control allocation: [[10.0]]                        ┃                   ║\n ║  ┃                                                      ┃                   ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛                   ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\nocomp: Executed 2 of 2 trials\n    icomp: Executed 1 of 1 trial (depth: 1)\n    icomp: Executed 1 of 1 trial (depth: 1)\n'
        assert actual_output == expected_output

        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp BEFORE its Trial 20 ━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [44.0]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                    │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                       │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 42.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -11.0                   │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -11. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -11.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -11.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 33.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-11.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 24.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                            ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 20  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                     │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                   │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -20. (monitored by iController      │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -20.0                        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                                  │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 24.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-20.0]]                                      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: -20.0                      │ │                             ┃ ║\n ║  ┃  │ │ output: 24.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-20.0]]                                                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━  oController SIMULATION OF ocomp AFTER its Trial 0 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[-2.0]]                                                       ┃ ║\n ║  ┃  outcome: [24.0]                                                       ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp BEFORE its Trial━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [24.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 22.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 13.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 4.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                    │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 22.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -2.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 22.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-2.0]]                                                 ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp BEFORE its Trial━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [24.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 22.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 13.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 4.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 4.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 4.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[1.0]]                                           ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 1  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1]]                                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp BEFORE its Trial 21 ━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [24.0]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 1.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                                │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 1. (monitored by iController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 1.0                     │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 1.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                        │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 25.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[1.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 5.5                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 5.5                                │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 5.5 (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 5.5                     │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 5.5                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 5.5                        │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 29.5                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[5.5]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 10.0                    │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 10.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 10. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 10.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 10.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 10.0                       │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 34.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[10.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[10.0]]                           ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 21  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 1.0                      │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                    │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: 10. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: 10.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 34.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[10.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: 10.0                       │ │                             ┃ ║\n ║  ┃  │ │ output: 34.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[10.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━  oController SIMULATION OF ocomp AFTER its Trial 1 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[1.0]]                                                        ┃ ║\n ║  ┃  outcome: [34.0]                                                       ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp BEFORE its Trial━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [24.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 22.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 13.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 4.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                    │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 22.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -2.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 22.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-2.0]]                                                 ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp BEFORE its Trial━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [24.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 22.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 13.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 4.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 4.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 4.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[10.0]]                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\nocomp: Executed 2 of 2 trials\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n'
        assert actual_output == expected_output

        icomp.controller_mode = pnl.AFTER
        ocomp.controller_mode = pnl.BEFORE

        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━  oController SIMULATION OF ocomp BEFORE its Trial 0 ━┓                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  state: [[-2.0]]                                      ┃                  ║\n ║  ┃  outcome: [34.0]                                      ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  control allocation: [[1.0]]                          ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛                  ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 22  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                     │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                   │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -20. (monitored by iController      │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -20.0                        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                                  │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 14.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-20.0]]                                      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp AFTER its Trial 22 ━┓   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                      ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [14.0]                                      ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                          ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛   ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: -20.0                      │ │                             ┃ ║\n ║  ┃  │ │ output: 14.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-20.0]]                                                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━  oController SIMULATION OF ocomp BEFORE its Trial 1 ━┓                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  state: [[1.0]]                                       ┃                  ║\n ║  ┃  outcome: [14.0]                                      ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  control allocation: [[10.0]]                         ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛                  ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 1  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1]]                                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 23  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 1.0                      │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                    │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: 10. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: 10.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 24.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[10.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp AFTER its Trial 23 ━┓   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[1.0]]                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [24.0]                                      ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[10.0]]                         ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                       ┃   ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛   ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: 10.0                       │ │                             ┃ ║\n ║  ┃  │ │ output: 24.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[10.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\nocomp: Executed 2 of 2 trials\n    icomp: Executed 1 of 1 trial (depth: 1)\n    icomp: Executed 1 of 1 trial (depth: 1)\n'
        assert actual_output == expected_output

        ocomp.run(inputs=inputs_dict,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.ON,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━  oController SIMULATION OF ocomp BEFORE its Trial 0 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[-2.0]]                                                       ┃ ║\n ║  ┃  outcome: [24.0]                                                       ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 4.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [4.0]                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -7.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -16.0                     │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 4.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -200.0                  │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -200.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -200. (monitored by            │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController Objective Mechanism and   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ oController Objective Mechanism)      │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -200.0                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -200.0                            │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -200.0                     │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -176.0                    │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-200.0]]                                ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-176.0]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -178.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -187.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -196.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -200.0                     │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -176.0                    │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-200.0]]                                               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[1.0]]                                           ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 24  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                     │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                   │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -20. (monitored by iController      │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -20.0                        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -20.0                                  │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -20.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 4.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-20.0]]                                      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━  iController SIMULATION OF icomp AFTER its Trial 24 ━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [4.0]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                    │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                       │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 2.0                       │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -11.0                   │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -11. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -11.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -11.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -7.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-11.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -16.0                     │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                            ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: -20.0                      │ │                             ┃ ║\n ║  ┃  │ │ output: 4.0                       │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-20.0]]                                                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━  oController SIMULATION OF ocomp BEFORE its Trial 1 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[1.0]]                                                        ┃ ║\n ║  ┃  outcome: [4.0]                                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                   │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                      │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: 4.0                       │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [4.0]                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: 2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -7.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -16.0                     │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                      │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: 4.0                       │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                     │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -200.0                  │ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                     │           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -200.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -200. (monitored by            │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController Objective Mechanism and   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ oController Objective Mechanism)      │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -200.0                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -200.0                            │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -200.0                     │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -176.0                    │ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                       │         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-200.0]]                                ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-176.0]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                    │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                       │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -178.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -11. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -11.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -11.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -11.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -187.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-11.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌───────────  Time Step 0 ────────────┐      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭────────────── ia ───────────────╮ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                     │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0          │ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                   │ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰─────────────────────────────────╯ │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                     │      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └─────────────────────────────────────┘      ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                      │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -196.0                    │ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰───────────────────────────────────╯ │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                       │    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └───────────────────────────────────────┘    ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────  Time Step 1 ─────────────┐                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╭─ oController Objective Mechanism ─╮ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -200.0                     │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -176.0                    │ │                        ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────╯ │                        ┃ ┃ ║\n ║  ┃  ┃  │                                       │                        ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────┘                        ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-200.0]]                                               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[10.0]]                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 1  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1]]                                                          ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 25  ━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 1.0                      │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                    │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: 10. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: 10.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 10.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: 10.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: 14.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[10.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━  iController SIMULATION OF icomp AFTER its Trial 25 ━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[1.0]]                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [14.0]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 1.0                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                                │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 1. (monitored by iController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 1.0                     │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 1.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                        │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 15.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[1.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 5.5                     │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 5.5                                │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 5.5 (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 5.5                     │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 5.5                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 5.5                        │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 19.5                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[5.5]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 2: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[1.0]]                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌───────────  Time Step 0 ────────────┐           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭────────────── ia ───────────────╮ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 1.0                      │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭────────── params ───────────╮ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5 │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0          │ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰─────────────────────────────╯ │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 10.0                    │ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰─────────────────────────────────╯ │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                     │           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └─────────────────────────────────────┘           ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 10.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: 10. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: 10.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 10.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────  Time Step 2 ─────────────┐         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─ iController Objective Mechanism ─╮ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: 10.0                       │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: 24.0                      │ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────╯ │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                       │         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────┘         ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[10.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[10.0]]                           ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: 10.0                       │ │                             ┃ ║\n ║  ┃  │ │ output: 14.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[10.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\nocomp: Executed 2 of 2 trials\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n    ocomp: Simulated 2 trials (depth: 1)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n      icomp: Executed 1 of 1 trial (depth: 2)\n        icomp: Simulated 3 trials (depth: 3)\n    icomp: Executed 1 of 1 trial (depth: 1)\n      icomp: Simulated 3 trials (depth: 2)\n'
        assert actual_output == expected_output

    def test_nested_comps_and_sims_with_modulated_and_monitored_params_and_use_prefs(self):

        # instantiate mechanisms and inner comp
        ia = pnl.TransferMechanism(name='ia')
        ib = pnl.TransferMechanism(name='ib')
        ia.reportOutputPref=[pnl.INTERCEPT]
        ib.reportOutputPref=[pnl.VALUE, pnl.VARIABLE]
        icomp = pnl.Composition(name='icomp', controller_mode=pnl.AFTER)

        # set up structure of inner comp
        icomp.add_node(ia, required_roles=pnl.NodeRole.INPUT)
        icomp.add_node(ib, required_roles=pnl.NodeRole.OUTPUT)
        icomp.add_projection(pnl.MappingProjection(), sender=ia, receiver=ib)

        # add controller to inner comp
        icomp.add_controller(
                pnl.OptimizationControlMechanism(
                        agent_rep=icomp,
                        state_features=[ia.input_port],
                        name="iController",
                        objective_mechanism=pnl.ObjectiveMechanism(
                                monitor=ib.output_port,
                                function=pnl.SimpleIntegrator,
                                name="iController Objective Mechanism"
                        ),
                        function=pnl.GridSearch(direction=pnl.MAXIMIZE),
                        control_signals=[pnl.ControlSignal(projections=[(pnl.SLOPE, ia)],
                                                           variable=1.0,
                                                           intensity_cost_function=pnl.Linear(slope=0.0),
                                                           allocation_samples=pnl.SampleSpec(start=1.0,
                                                                                             stop=10.0,
                                                                                             num=2))])
        )

        # instantiate outer comp
        ocomp = pnl.Composition(name='ocomp', controller_mode=pnl.BEFORE)

        # setup structure of outer comp
        ocomp.add_node(icomp)

        # ocomp._analyze_graph()

        # add controller to outer comp
        ocomp.add_controller(
                pnl.OptimizationControlMechanism(
                        agent_rep=ocomp,
                        state_features=[ia.input_port],
                        # state_features=[ocomp.input_CIM.output_ports[0]],
                        name="oController",
                        objective_mechanism=pnl.ObjectiveMechanism(
                                monitor=ib.output_port,
                                function=pnl.SimpleIntegrator,
                                name="oController Objective Mechanism"
                        ),
                        function=pnl.GridSearch(direction=pnl.MAXIMIZE),
                        control_signals=[pnl.ControlSignal(projections=[(pnl.SLOPE, ia)],
                                                           variable=1.0,
                                                           intensity_cost_function=pnl.Linear(slope=0.0),
                                                           allocation_samples=pnl.SampleSpec(start=1.0,
                                                                                             stop=10.0,
                                                                                             num=3))])
        )

        # inputs_dict = {icomp:{ia: [[-2], [1]]}}

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.FULL,
                  report_params=[pnl.ReportParams.MODULATED, pnl.ReportParams.MONITORED],
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  # report_to_devices=ReportDevices.RECORD
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━  oController SIMULATION OF ocomp BEFORE its Trial 0 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[-2.0]]                                                       ┃ ║\n ║  ┃  outcome: [0.0]                                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -2. (monitored by iController) │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-2.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -4. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -4.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -22. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -22.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐                ┃ ┃ ║\n ║  ┃  ┃  │                                               │                ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ─────╮ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -2.0                               │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -2. (monitored by oController) │ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                       │ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -2.0                              │ │                ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │                ┃ ┃ ║\n ║  ┃  ┃  │                                               │                ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────────────┘                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-2.0]]                                                 ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -11. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -11.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -11. (monitored by             │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-11.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-11.0]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -13. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -13.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -31. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -31.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌─────────────────  Time Step 1 ─────────────────┐               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ──────╮ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -11.0                               │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭──────────────── params ────────────────╮ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -11. (monitored by oController) │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                        │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰────────────────────────────────────────╯ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -11.0                              │ │               ┃ ┃ ║\n ║  ┃  ┃  │ ╰────────────────────────────────────────────╯ │               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  └────────────────────────────────────────────────┘               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-11.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 2: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by             │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-20.0]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -22. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -22.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -40. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -40.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌─────────────────  Time Step 1 ─────────────────┐               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ──────╮ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                               │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭──────────────── params ────────────────╮ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -20. (monitored by oController) │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                        │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰────────────────────────────────────────╯ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -20.0                              │ │               ┃ ┃ ║\n ║  ┃  ┃  │ ╰────────────────────────────────────────────╯ │               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  └────────────────────────────────────────────────┘               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[1.0]]                                           ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 0 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ia ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5                │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     slope: 1.0 (modulated by iController   │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ and oController)                           │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -2. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -2.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────────  Time Step 2 ─────────────────┐      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                               │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭───── iController Objective Mechanism ─────╮ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                               │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭─────────────── params ────────────────╮ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -2. (monitored by iController) │ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │                                       │ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰───────────────────────────────────────╯ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                              │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────────────╯ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                               │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────────────┘      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-2.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━  iController SIMULATION OF icomp AFTER its Trial 0 ━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [-2.0]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -4. (monitored by iController) │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -4.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -22. (monitored by             │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -22.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                            ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────────  Time Step 1 ─────────────────┐                     ┃ ║\n ║  ┃  │                                               │                     ┃ ║\n ║  ┃  │ ╭───── oController Objective Mechanism ─────╮ │                     ┃ ║\n ║  ┃  │ │ input: -2.0                               │ │                     ┃ ║\n ║  ┃  │ │ ╭─────────────── params ────────────────╮ │ │                     ┃ ║\n ║  ┃  │ │ │ value: -2. (monitored by oController) │ │ │                     ┃ ║\n ║  ┃  │ │ │                                       │ │ │                     ┃ ║\n ║  ┃  │ │ ╰───────────────────────────────────────╯ │ │                     ┃ ║\n ║  ┃  │ │ output: -2.0                              │ │                     ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────╯ │                     ┃ ║\n ║  ┃  │                                               │                     ┃ ║\n ║  ┃  └───────────────────────────────────────────────┘                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-2.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.FULL,
                  # report_params=[ReportParams.MODULATED, ReportParams.MONITORED],
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.OFF,
                  report_to_devices=ReportDevices.DIVERT
                  # report_to_devices=ReportDevices.RECORD
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━  oController SIMULATION OF ocomp BEFORE its Trial 0 ━┓                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  state: [[-2.0]]                                      ┃                  ║\n ║  ┃  outcome: [-2.0]                                      ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┃  control allocation: [[1.0]]                          ┃                  ║\n ║  ┃                                                       ┃                  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛                  ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 1  ━━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────  Time Step 0 ────────────┐                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────── ia ───────────────╮ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                     │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────── params ───────────╮ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5 │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0          │ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰─────────────────────────────╯ │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                    │ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰─────────────────────────────────╯ │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                     │                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └─────────────────────────────────────┘                ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -2. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -2.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────  Time Step 2 ─────────────┐              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭─ iController Objective Mechanism ─╮ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                       │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -4.0                      │ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────╯ │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                       │              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────┘              ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-2.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━  iController SIMULATION OF icomp AFTER its Trial 1 ━┓    ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                      ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                     ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [-4.0]                                     ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                      ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                         ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                      ┃    ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛    ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────  Time Step 1 ─────────────┐                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  │ ╭─ oController Objective Mechanism ─╮ │                             ┃ ║\n ║  ┃  │ │ input: -2.0                       │ │                             ┃ ║\n ║  ┃  │ │ output: -4.0                      │ │                             ┃ ║\n ║  ┃  │ ╰───────────────────────────────────╯ │                             ┃ ║\n ║  ┃  │                                       │                             ┃ ║\n ║  ┃  └───────────────────────────────────────┘                             ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-2.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\n'
        assert actual_output == expected_output

        ocomp.run(inputs={icomp:-2},
                  report_output=ReportOutput.ON,
                  # report_params=[ReportParams.MODULATED, ReportParams.MONITORED],
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  # report_to_devices=ReportDevices.RECORD
                  )
        actual_output = ocomp.rich_diverted_reports
        # FIX: THIS ACCOMODATES AN ERROR IN WHICH A NESTED COMPOSITION CONTINUES TO INCREMENT ITS TRIAL COUNT
        #       OVER SEQUENTIAL EXECUTIONS OF THE OUTER COMPOSITION (SINCE IT IS CALLED USING ITS execute() METHOD
        #       (RATHER THAN ITS RUN METHOD, THAT WOULD RESET ITS COUNT ON EACH CALL).
        expected_output = '\nExecution of ocomp:\n  oController simulation of ocomp before TRIAL 0\n      ocomp SIMULATION 0 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 1 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n      ocomp SIMULATION 2 TRIAL 0 ====================\n      Time Step 0 ---------\n      Execution of icomp within ocomp:\n        icomp TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n      iController simulation of icomp after TRIAL 0\n          icomp SIMULATION 0 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n          icomp SIMULATION 1 TRIAL 0 ====================\n          Time Step 0 ---------\n            ia executed\n          Time Step 1 ---------\n            ib executed\n          Time Step 2 ---------\n            iController Objective Mechanism executed\n      Time Step 1 ---------\n        oController Objective Mechanism executed\n    ocomp TRIAL 0 ====================\n    Time Step 0 ---------\n    Execution of icomp within ocomp:\n      icomp TRIAL 2 ====================\n      Time Step 0 ---------\n        ia executed\n      Time Step 1 ---------\n        ib executed\n      Time Step 2 ---------\n        iController Objective Mechanism executed\n    iController simulation of icomp after TRIAL 2\n        icomp SIMULATION 0 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n        icomp SIMULATION 1 TRIAL 0 ====================\n        Time Step 0 ---------\n          ia executed\n        Time Step 1 ---------\n          ib executed\n        Time Step 2 ---------\n          iController Objective Mechanism executed\n    Time Step 1 ---------\n      oController Objective Mechanism executed\n'
        assert actual_output == expected_output

        ocomp.execute(inputs={icomp:-2},
                  report_output=ReportOutput.FULL,
                  report_params=[pnl.ReportParams.MODULATED, pnl.ReportParams.MONITORED],
                  report_progress=ReportProgress.OFF,
                  report_simulations=ReportSimulations.ON,
                  report_to_devices=ReportDevices.DIVERT
                  # report_to_devices=ReportDevices.RECORD
                  )
        actual_output = ocomp.rich_diverted_reports
        expected_output = '\n ╔════════════════════════════ EXECUTION OF ocomp  ════════════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━  oController SIMULATION OF ocomp BEFORE its Trial 0 ━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  state: [[-2.0]]                                                       ┃ ║\n ║  ┃  outcome: [-6.0]                                                       ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 0: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -8. (monitored by iController) │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -8.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-8.0]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -10. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -10.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -28. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -28.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐                ┃ ┃ ║\n ║  ┃  ┃  │                                               │                ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ─────╮ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -2.0                               │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -8. (monitored by oController) │ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                       │ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │                ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -8.0                              │ │                ┃ ┃ ║\n ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │                ┃ ┃ ║\n ║  ┃  ┃  │                                               │                ┃ ┃ ║\n ║  ┃  ┃  └───────────────────────────────────────────────┘                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-2.0]]                                                 ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 1: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -11. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -11.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -11.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -11.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -17. (monitored by             │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -17.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-11.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-17.0]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -19. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -19.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -37. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -37.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌─────────────────  Time Step 1 ─────────────────┐               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ──────╮ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -11.0                               │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭──────────────── params ────────────────╮ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -17. (monitored by oController) │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                        │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰────────────────────────────────────────╯ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -17.0                              │ │               ┃ ┃ ║\n ║  ┃  ┃  │ ╰────────────────────────────────────────────╯ │               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  └────────────────────────────────────────────────┘               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-11.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┏━━━━━━━━━━━━━━━━━━  ocomp SIMULATION 2: Trial 0  ━━━━━━━━━━━━━━━━━━┓ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  input: [[-2.0]]                                                  ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌────────────────────────  Time Step 0 ────────────────────────┐ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  │  ┏━━━━━━━━━━━━ EXECUTION OF icomp within ocomp ━━━━━━━━━━━━┓ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━━━━━━━━━━━━━━━━━  icomp: Trial 0  ━━━━━━━━━━━━━━━━━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ value: -26. (monitored by             │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ │ output: -26.0                             │ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  │                                               │ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┏━  iController SIMULATION OF icomp AFTER its Trial ━┓ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  state: [[-2.0]]                                   ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  outcome: [-26.0]                                  ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -2. (monitored by         │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -2.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -2.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -28. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -28.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-2.0]]                             ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┏━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━┓ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  input: [[-2.0]]                              ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 0 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ia ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -2.0                          │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-5      │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     intercept: 0.0               │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     slope: 1.0 (modulated by     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController and oController)     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 1 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭───────────────── ib ─────────────────╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -20. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController Objective Mechanism  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ and oController Objective        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ Mechanism)                       │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ function: Linear Function-14     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │     variable: -20.0              │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -20.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  ┌──────────────  Time Step 2 ──────────────┐ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╭── iController Objective Mechanism ───╮ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ input: -20.0                         │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╭───────────── params ─────────────╮ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ value: -46. (monitored by        │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │ iController)                     │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ │                                  │ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ ╰──────────────────────────────────╯ │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ │ output: -46.0                        │ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │ ╰──────────────────────────────────────╯ │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  │                                          │ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  └──────────────────────────────────────────┘ ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃  result: [[-20.0]]                            ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┃                                               ┃ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃  control allocation: [[1.0]]                       ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┃                                                    ┃ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┃                                                         ┃ │ ┃ ┃ ║\n ║  ┃  ┃  │  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ │ ┃ ┃ ║\n ║  ┃  ┃  │                                                              │ ┃ ┃ ║\n ║  ┃  ┃  └──────────────────────────────────────────────────────────────┘ ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  ┌─────────────────  Time Step 1 ─────────────────┐               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  │ ╭───── oController Objective Mechanism ──────╮ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ input: -20.0                               │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╭──────────────── params ────────────────╮ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │ value: -26. (monitored by oController) │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ │                                        │ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ ╰────────────────────────────────────────╯ │ │               ┃ ┃ ║\n ║  ┃  ┃  │ │ output: -26.0                              │ │               ┃ ┃ ║\n ║  ┃  ┃  │ ╰────────────────────────────────────────────╯ │               ┃ ┃ ║\n ║  ┃  ┃  │                                                │               ┃ ┃ ║\n ║  ┃  ┃  └────────────────────────────────────────────────┘               ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┃  result: [[-20.0]]                                                ┃ ┃ ║\n ║  ┃  ┃                                                                   ┃ ┃ ║\n ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  control allocation: [[1.0]]                                           ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━━━━━━━━━  ocomp: Trial 0  ━━━━━━━━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[-2]]                                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 0 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │  ╔══════════════ EXECUTION OF icomp within ocomp ═══════════════╗ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━━━━━━━━━━━━━━━━━━  icomp: Trial 3  ━━━━━━━━━━━━━━━━━━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  input: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 0 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ia ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-5                │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     intercept: 0.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     slope: 1.0 (modulated by iController   │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ and oController)                           │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌───────────────────  Time Step 1 ───────────────────┐ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭────────────────────── ib ──────────────────────╮ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                                    │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭────────────────── params ──────────────────╮ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -2. (monitored by iController       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism and oController        │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ Objective Mechanism)                       │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ function: Linear Function-14               │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │     variable: -2.0                         │ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰────────────────────────────────────────────╯ │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -2.0                                   │ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰────────────────────────────────────────────────╯ │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                                    │ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └────────────────────────────────────────────────────┘ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┌────────────────  Time Step 2 ─────────────────┐      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                               │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╭───── iController Objective Mechanism ─────╮ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ input: -2.0                               │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╭─────────────── params ────────────────╮ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │ value: -8. (monitored by iController) │ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ │                                       │ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ ╰───────────────────────────────────────╯ │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ │ output: -8.0                              │ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │ ╰───────────────────────────────────────────╯ │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  │                                               │      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  └───────────────────────────────────────────────┘      ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  result: [[-2.0]]                                       ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ║  ┏━━  iController SIMULATION OF icomp AFTER its Trial 3 ━━━┓ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  state: [[-2.0]]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  outcome: [-8.0]                                        ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 0: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -2. (monitored by iController  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -2.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -2.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -10. (monitored by             │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -10.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-2.0]]                                  ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┏━━━━━━━━━━  icomp SIMULATION 1: Trial 0  ━━━━━━━━━━━┓ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  input: [[-2.0]]                                   ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 0 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ia ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -2.0                               │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-5           │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     intercept: 0.0                    │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     slope: 1.0 (modulated by          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController and oController)          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 1 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭─────────────────── ib ────────────────────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -20. (monitored by iController │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism and oController   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ Objective Mechanism)                  │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ function: Linear Function-14          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │     variable: -20.0                   │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -20.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  ┌────────────────  Time Step 2 ─────────────────┐ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╭───── iController Objective Mechanism ─────╮ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ input: -20.0                              │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╭─────────────── params ────────────────╮ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ value: -28. (monitored by             │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │ iController)                          │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ │                                       │ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ ╰───────────────────────────────────────╯ │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ │ output: -28.0                             │ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │ ╰───────────────────────────────────────────╯ │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  │                                               │ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  └───────────────────────────────────────────────┘ ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃  result: [[-20.0]]                                 ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┃                                                    ┃ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃  control allocation: [[1.0]]                            ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┃                                                         ┃ ║ │ ┃ ║\n ║  ┃  │  ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║ │ ┃ ║\n ║  ┃  │  ║                                                              ║ │ ┃ ║\n ║  ┃  │  ╚══════════════════════════════════════════════════════════════╝ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────────────────  Time Step 1 ─────────────────┐                     ┃ ║\n ║  ┃  │                                               │                     ┃ ║\n ║  ┃  │ ╭───── oController Objective Mechanism ─────╮ │                     ┃ ║\n ║  ┃  │ │ input: -2.0                               │ │                     ┃ ║\n ║  ┃  │ │ ╭─────────────── params ────────────────╮ │ │                     ┃ ║\n ║  ┃  │ │ │ value: -8. (monitored by oController) │ │ │                     ┃ ║\n ║  ┃  │ │ │                                       │ │ │                     ┃ ║\n ║  ┃  │ │ ╰───────────────────────────────────────╯ │ │                     ┃ ║\n ║  ┃  │ │ output: -8.0                              │ │                     ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────╯ │                     ┃ ║\n ║  ┃  │                                               │                     ┃ ║\n ║  ┃  └───────────────────────────────────────────────┘                     ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[-2.0]]                                                      ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\n'
        assert actual_output == expected_output

        icomp.controller.reportOutputPref = ReportOutput.ON
        ocomp.controller.reportOutputPref = ReportOutput.FULL


    @pytest.mark.pytorch
    def test_autodiff_report(self):
        xor_in = pnl.TransferMechanism(name='xor_in',
                                       default_variable=np.zeros(2))

        xor_hid = pnl.TransferMechanism(name='xor_hid',
                                        default_variable=np.zeros(10),
                                        function=pnl.Logistic())

        xor_out = pnl.TransferMechanism(name='xor_out',
                                        default_variable=np.zeros(1),
                                        function=pnl.Logistic())

        hid_map = pnl.MappingProjection()
        out_map = pnl.MappingProjection()

        xor = pnl.AutodiffComposition()

        xor.add_node(xor_in)
        xor.add_node(xor_hid)
        xor.add_node(xor_out)

        xor.add_projection(sender=xor_in, projection=hid_map, receiver=xor_hid)
        xor.add_projection(sender=xor_hid, projection=out_map, receiver=xor_out)

        xor_inputs = np.array([[0, 0],[0, 1],[1, 0],[1, 1]])
        xor_targets = np.array([[0],[1],[1],[0]])

        training_inputs = {"inputs": {xor_in:xor_inputs},
                          "targets": {xor_out:xor_targets},
                          "epochs": 3}


        xor.learn(inputs= training_inputs,
                  report_output=ReportOutput.OFF,
                  report_progress=ReportProgress.ON,
                  report_to_devices=ReportDevices.DIVERT)
        actual_report = xor.rich_diverted_reports
        expected_report = 'autodiff_composition: Trained 1 trial\nautodiff_composition: Trained 2 trials\nautodiff_composition: Trained 3 trials\nautodiff_composition: Trained 4 trials\nautodiff_composition: Trained 5 trials\nautodiff_composition: Trained 6 trials\nautodiff_composition: Trained 7 trials\nautodiff_composition: Trained 8 trials\nautodiff_composition: Trained 9 trials\nautodiff_composition: Trained 10 trials\nautodiff_composition: Trained 11 trials\nautodiff_composition: Trained 12 trials\n'
        assert actual_report == expected_report

        xor.run(inputs={xor_in:xor_inputs},
                report_output=ReportOutput.OFF,
                report_progress=ReportProgress.ON,
                report_to_devices=ReportDevices.DIVERT
                )
        actual_report = xor.rich_diverted_reports
        expected_report = '\'autodiff_composition: Executed 1 of 4 trials\\nautodiff_composition: Executed 2 of 4 trials\\nautodiff_composition: Executed 3 of 4 trials\\nautodiff_composition: Executed 4 of 4 trials\\n\''
        assert repr(actual_report) == expected_report

        xor.learn(inputs= training_inputs,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.ON,
                  report_to_devices=ReportDevices.DIVERT
                  )
        actual_report = xor.rich_diverted_reports
        expected_report = '\nExecution of autodiff_composition:\n    autodiff_composition TRAINING TRIAL 0 ====================\n    autodiff_composition TRAINING TRIAL 1 ====================\n    autodiff_composition TRAINING TRIAL 2 ====================\n    autodiff_composition TRAINING TRIAL 3 ====================\n    autodiff_composition TRAINING TRIAL 4 ====================\n    autodiff_composition TRAINING TRIAL 5 ====================\n    autodiff_composition TRAINING TRIAL 6 ====================\n    autodiff_composition TRAINING TRIAL 7 ====================\n    autodiff_composition TRAINING TRIAL 8 ====================\n    autodiff_composition TRAINING TRIAL 9 ====================\n    autodiff_composition TRAINING TRIAL 10 ====================\n    autodiff_composition TRAINING TRIAL 11 ====================\nautodiff_composition: Trained 12 trials\n'
        assert actual_report == expected_report

        xor.run(inputs={xor_in:xor_inputs},
                report_output=ReportOutput.ON,
                report_progress=ReportProgress.ON,
                report_to_devices=ReportDevices.DIVERT
                )
        actual_report = xor.rich_diverted_reports
        expected_report = '\nExecution of autodiff_composition:\n    autodiff_composition TRIAL 0 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 1 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 2 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 3 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\nautodiff_composition: Executed 4 of 4 trials\n'
        assert actual_report == expected_report

        xor.learn(inputs=training_inputs,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.ON,
                  report_to_devices=ReportDevices.DIVERT)
        actual_report = xor.rich_diverted_reports
        expected_report = '\n ╔══ EXECUTION OF autodiff_composition  ═══╗\n ║                                         ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 0  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933057795354014]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 1  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.999331787548446]]    ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 2  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993317875516309]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 3  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9998504229552773]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 4  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933055512239266]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 5  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993317539824547]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 6  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993317539856401]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 7  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9998504138991838]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 8  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933053228968025]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 9  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993317204136144]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 10  ━┓ ║\n ║  ┃                                    ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]            ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┃  result: [[0.9993317204168003]]    ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 11  ━┓ ║\n ║  ┃                                    ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]            ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┃  result: [[0.999850404842228]]     ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                         ║\n ╚═════════════════════════════════════════╝\n\nautodiff_composition: Trained 12 trials\n'
        assert actual_report == expected_report

        xor.run(inputs={xor_in:xor_inputs},
                report_output=ReportOutput.FULL,
                report_progress=ReportProgress.ON,
                report_to_devices=ReportDevices.DIVERT
                )
        actual_report = xor.rich_diverted_reports
        expected_report = '\n ╔════════════════════ EXECUTION OF autodiff_composition  ═════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━  autodiff_composition: Trial 0  ━━━━━━━━━━━━━┓               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  input: [[0, 0], [0.5]]                                  ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌────  Time Step 0 ────┐                                ┃               ║\n ║  ┃  │                      │                                ┃               ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                ┃               ║\n ║  ┃  │ │ input: [[0. 0.]] │ │                                ┃               ║\n ║  ┃  │ │ output: 0.0 0.0  │ │                                ┃               ║\n ║  ┃  │ ╰──────────────────╯ │                                ┃               ║\n ║  ┃  │                      │                                ┃               ║\n ║  ┃  └──────────────────────┘                                ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌───────────────────  Time Step 1 ────────────────────┐ ┃               ║\n ║  ┃  │                                                     │ ┃               ║\n ║  ┃  │ ╭──────────────────── xor_hid ────────────────────╮ │ ┃               ║\n ║  ┃  │ │ input: [[0. 0. 0. 0. 0. 0. 0. 0. 0. 0.]]        │ │ ┃               ║\n ║  ┃  │ │ output: 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 │ │ ┃               ║\n ║  ┃  │ ╰─────────────────────────────────────────────────╯ │ ┃               ║\n ║  ┃  │                                                     │ ┃               ║\n ║  ┃  └─────────────────────────────────────────────────────┘ ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌──  Time Step 2 ───┐                                   ┃               ║\n ║  ┃  │                   │                                   ┃               ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                   ┃               ║\n ║  ┃  │ │ input: 5.0    │ │                                   ┃               ║\n ║  ┃  │ │ output: 0.993 │ │                                   ┃               ║\n ║  ┃  │ ╰───────────────╯ │                                   ┃               ║\n ║  ┃  │                   │                                   ┃               ║\n ║  ┃  └───────────────────┘                                   ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  result: [[0.9933050945540283]]                          ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛               ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 1  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[0, 1], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[0. 1.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 0.0 1.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[0.99999972 0.99999972 0.99999972 0.99999972          │ │ ┃ ║\n ║  ┃  │ │ 0.99999972 0.99999972                                         │ │ ┃ ║\n ║  ┃  │ │   0.99999972 0.99999972 0.99999972 0.99999972]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 │ │ ┃ ║\n ║  ┃  │ │ 0.731                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──  Time Step 2 ───┐                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                                 ┃ ║\n ║  ┃  │ │ input: 7.31   │ │                                                 ┃ ║\n ║  ┃  │ │ output: 0.999 │ │                                                 ┃ ║\n ║  ┃  │ ╰───────────────╯ │                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  └───────────────────┘                                                 ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.9993317190927715]]                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 2  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1, 0], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[1. 0.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 1.0 0.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[0.99999972 0.99999972 0.99999972 0.99999972          │ │ ┃ ║\n ║  ┃  │ │ 0.99999972 0.99999972                                         │ │ ┃ ║\n ║  ┃  │ │   0.99999972 0.99999972 0.99999972 0.99999972]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 │ │ ┃ ║\n ║  ┃  │ │ 0.731                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──  Time Step 2 ───┐                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                                 ┃ ║\n ║  ┃  │ │ input: 7.31   │ │                                                 ┃ ║\n ║  ┃  │ │ output: 0.999 │ │                                                 ┃ ║\n ║  ┃  │ ╰───────────────╯ │                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  └───────────────────┘                                                 ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.9993317190927715]]                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 3  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[1. 1.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 1.0 1.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[1.99999944 1.99999944 1.99999944 1.99999944          │ │ ┃ ║\n ║  ┃  │ │ 1.99999944 1.99999944                                         │ │ ┃ ║\n ║  ┃  │ │   1.99999944 1.99999944 1.99999944 1.99999944]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.881 0.881 0.881 0.881 0.881 0.881 0.881 0.881 0.881 │ │ ┃ ║\n ║  ┃  │ │ 0.881                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐                                                   ┃ ║\n ║  ┃  │                 │                                                   ┃ ║\n ║  ┃  │ ╭── xor_out ──╮ │                                                   ┃ ║\n ║  ┃  │ │ input: 8.81 │ │                                                   ┃ ║\n ║  ┃  │ │ output: 1.0 │ │                                                   ┃ ║\n ║  ┃  │ ╰─────────────╯ │                                                   ┃ ║\n ║  ┃  │                 │                                                   ┃ ║\n ║  ┃  └─────────────────┘                                                   ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.9998504044852918]]                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\nautodiff_composition: Executed 4 of 4 trials\n'
        assert actual_report == expected_report

        xor.learn(inputs= training_inputs,
                  report_output=ReportOutput.ON,
                  report_progress=ReportProgress.OFF,
                  report_to_devices=ReportDevices.DIVERT)
        actual_report = xor.rich_diverted_reports
        expected_report = '\nExecution of autodiff_composition:\n    autodiff_composition TRAINING TRIAL 0 ====================\n    autodiff_composition TRAINING TRIAL 1 ====================\n    autodiff_composition TRAINING TRIAL 2 ====================\n    autodiff_composition TRAINING TRIAL 3 ====================\n    autodiff_composition TRAINING TRIAL 4 ====================\n    autodiff_composition TRAINING TRIAL 5 ====================\n    autodiff_composition TRAINING TRIAL 6 ====================\n    autodiff_composition TRAINING TRIAL 7 ====================\n    autodiff_composition TRAINING TRIAL 8 ====================\n    autodiff_composition TRAINING TRIAL 9 ====================\n    autodiff_composition TRAINING TRIAL 10 ====================\n    autodiff_composition TRAINING TRIAL 11 ====================\n'
        assert actual_report == expected_report

        xor.run(inputs={xor_in:xor_inputs},
                report_output=ReportOutput.ON,
                report_progress=ReportProgress.OFF,
                report_to_devices=ReportDevices.DIVERT
                )
        actual_report = xor.rich_diverted_reports
        expected_report = '\nExecution of autodiff_composition:\n    autodiff_composition TRIAL 0 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 1 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 2 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n    autodiff_composition TRIAL 3 ====================\n    Time Step 0 ---------\n      xor_in executed\n    Time Step 1 ---------\n      xor_hid executed\n    Time Step 2 ---------\n      xor_out executed\n    Time Step 3 ---------\n    Time Step 4 ---------\n    Time Step 5 ---------\n'
        assert actual_report == expected_report

        xor.learn(inputs=training_inputs,
                  report_output=ReportOutput.FULL,
                  report_progress=ReportProgress.OFF,
                  report_to_devices=ReportDevices.DIVERT)
        actual_report = xor.rich_diverted_reports
        expected_report = '\n ╔══ EXECUTION OF autodiff_composition  ═══╗\n ║                                         ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 0  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933044094317858]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 1  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993315861097587]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 2  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993315861129465]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 3  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9998503686057807]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 4  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933041810263933]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 5  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.999331552526669]]    ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 6  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993315525298574]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 7  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9998503595445122]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 8  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9933039526053421]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 9  ━┓  ║\n ║  ┃                                   ┃  ║\n ║  ┃  input: [[1, 1], [0.5]]           ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┃  result: [[0.9993315189407287]]   ┃  ║\n ║  ┃                                   ┃  ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛  ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 10  ━┓ ║\n ║  ┃                                    ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]            ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┃  result: [[0.9993315189439175]]    ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                         ║\n ║  ┏━  autodiff_composition: Trial 11  ━┓ ║\n ║  ┃                                    ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]            ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┃  result: [[0.9998503504823807]]    ┃ ║\n ║  ┃                                    ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                         ║\n ╚═════════════════════════════════════════╝\n\n'
        assert actual_report == expected_report

        xor.run(inputs={xor_in:xor_inputs},
                report_output=ReportOutput.FULL,
                report_progress=ReportProgress.OFF,
                report_to_devices=ReportDevices.DIVERT
                )
        actual_report = xor.rich_diverted_reports
        expected_report = '\n ╔════════════════════ EXECUTION OF autodiff_composition  ═════════════════════╗\n ║                                                                             ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━  autodiff_composition: Trial 0  ━━━━━━━━━━━━━┓               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  input: [[0, 0], [0.5]]                                  ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌────  Time Step 0 ────┐                                ┃               ║\n ║  ┃  │                      │                                ┃               ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                ┃               ║\n ║  ┃  │ │ input: [[0. 0.]] │ │                                ┃               ║\n ║  ┃  │ │ output: 0.0 0.0  │ │                                ┃               ║\n ║  ┃  │ ╰──────────────────╯ │                                ┃               ║\n ║  ┃  │                      │                                ┃               ║\n ║  ┃  └──────────────────────┘                                ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌───────────────────  Time Step 1 ────────────────────┐ ┃               ║\n ║  ┃  │                                                     │ ┃               ║\n ║  ┃  │ ╭──────────────────── xor_hid ────────────────────╮ │ ┃               ║\n ║  ┃  │ │ input: [[0. 0. 0. 0. 0. 0. 0. 0. 0. 0.]]        │ │ ┃               ║\n ║  ┃  │ │ output: 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 0.5 │ │ ┃               ║\n ║  ┃  │ ╰─────────────────────────────────────────────────╯ │ ┃               ║\n ║  ┃  │                                                     │ ┃               ║\n ║  ┃  └─────────────────────────────────────────────────────┘ ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  ┌──  Time Step 2 ───┐                                   ┃               ║\n ║  ┃  │                   │                                   ┃               ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                   ┃               ║\n ║  ┃  │ │ input: 5.0    │ │                                   ┃               ║\n ║  ┃  │ │ output: 0.993 │ │                                   ┃               ║\n ║  ┃  │ ╰───────────────╯ │                                   ┃               ║\n ║  ┃  │                   │                                   ┃               ║\n ║  ┃  └───────────────────┘                                   ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┃  result: [[0.9933037241686309]]                          ┃               ║\n ║  ┃                                                          ┃               ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛               ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 1  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[0, 1], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[0. 1.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 0.0 1.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[0.99999953 0.99999953 0.99999953 0.99999953          │ │ ┃ ║\n ║  ┃  │ │ 0.99999953 0.99999953                                         │ │ ┃ ║\n ║  ┃  │ │   0.99999953 0.99999953 0.99999953 0.99999953]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 │ │ ┃ ║\n ║  ┃  │ │ 0.731                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──  Time Step 2 ───┐                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                                 ┃ ║\n ║  ┃  │ │ input: 7.31   │ │                                                 ┃ ║\n ║  ┃  │ │ output: 0.999 │ │                                                 ┃ ║\n ║  ┃  │ ╰───────────────╯ │                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  └───────────────────┘                                                 ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.999331517619013]]                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 2  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1, 0], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[1. 0.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 1.0 0.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[0.99999953 0.99999953 0.99999953 0.99999953          │ │ ┃ ║\n ║  ┃  │ │ 0.99999953 0.99999953                                         │ │ ┃ ║\n ║  ┃  │ │   0.99999953 0.99999953 0.99999953 0.99999953]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 0.731 │ │ ┃ ║\n ║  ┃  │ │ 0.731                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──  Time Step 2 ───┐                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  │ ╭─── xor_out ───╮ │                                                 ┃ ║\n ║  ┃  │ │ input: 7.31   │ │                                                 ┃ ║\n ║  ┃  │ │ output: 0.999 │ │                                                 ┃ ║\n ║  ┃  │ ╰───────────────╯ │                                                 ┃ ║\n ║  ┃  │                   │                                                 ┃ ║\n ║  ┃  └───────────────────┘                                                 ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.999331517619013]]                                         ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ║  ┏━━━━━━━━━━━━━━━━━━━  autodiff_composition: Trial 3  ━━━━━━━━━━━━━━━━━━━━┓ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  input: [[1, 1], [0.5]]                                                ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌────  Time Step 0 ────┐                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  │ ╭───── xor_in ─────╮ │                                              ┃ ║\n ║  ┃  │ │ input: [[1. 1.]] │ │                                              ┃ ║\n ║  ┃  │ │ output: 1.0 1.0  │ │                                              ┃ ║\n ║  ┃  │ ╰──────────────────╯ │                                              ┃ ║\n ║  ┃  │                      │                                              ┃ ║\n ║  ┃  └──────────────────────┘                                              ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌──────────────────────────  Time Step 1 ───────────────────────────┐ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  │ ╭─────────────────────────── xor_hid ───────────────────────────╮ │ ┃ ║\n ║  ┃  │ │ input: [[1.99999906 1.99999906 1.99999906 1.99999906          │ │ ┃ ║\n ║  ┃  │ │ 1.99999906 1.99999906                                         │ │ ┃ ║\n ║  ┃  │ │   1.99999906 1.99999906 1.99999906 1.99999906]]               │ │ ┃ ║\n ║  ┃  │ │ output: 0.881 0.881 0.881 0.881 0.881 0.881 0.881 0.881 0.881 │ │ ┃ ║\n ║  ┃  │ │ 0.881                                                         │ │ ┃ ║\n ║  ┃  │ ╰───────────────────────────────────────────────────────────────╯ │ ┃ ║\n ║  ┃  │                                                                   │ ┃ ║\n ║  ┃  └───────────────────────────────────────────────────────────────────┘ ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  ┌─  Time Step 2 ──┐                                                   ┃ ║\n ║  ┃  │                 │                                                   ┃ ║\n ║  ┃  │ ╭── xor_out ──╮ │                                                   ┃ ║\n ║  ┃  │ │ input: 8.81 │ │                                                   ┃ ║\n ║  ┃  │ │ output: 1.0 │ │                                                   ┃ ║\n ║  ┃  │ ╰─────────────╯ │                                                   ┃ ║\n ║  ┃  │                 │                                                   ┃ ║\n ║  ┃  └─────────────────┘                                                   ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┃  result: [[0.9998503501251857]]                                        ┃ ║\n ║  ┃                                                                        ┃ ║\n ║  ┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛ ║\n ║                                                                             ║\n ╚═════════════════════════════════════════════════════════════════════════════╝\n\n'
        assert actual_report == expected_report
